/***************************** LICENSE START ***********************************

 Copyright 2013 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQFeatureType.h"

#include <QDebug>
#include <QDir>
#include <QFontMetrics>

#include "ConfigLoader.h"
#include "MvRequest.h"
#include "MvMiscelaneous.h"
#include "MvQMethods.h"
#include "MvQFeatureContextMenu.h"
#include "WmoWeatherSymbol.h"

QMap<QString, MvQFeatureGroupItem*> MvQFeatureGroupItem::items_;
QVector<MvQFeatureType*> MvQFeatureType::items_;
QStringList MvQFeatureType::groups_;
int MvQFeatureType::defaultPixSize_=20;
static QColor defaultPixBgColor(Qt::transparent);

//=================================================================
//
// MvQFeatureGroupItem
//
//=================================================================


MvQFeatureGroupItem::MvQFeatureGroupItem(const MvRequest& r)
{
    const char* verb = r.getVerb();
    Q_ASSERT(verb);
    Q_ASSERT(strcmp(verb, "item_group") == 0);

    if (const char* lCh = r("name")) {
        name_ = QString(lCh);
    }
    if (const char* lCh = r("label")) {
        label_ = QString(lCh);
    }
    if (const char* lCh = r("desc")) {
        desc_ = QString(lCh);
    }

    if (desc_.isEmpty()) {
        desc_ = label_;
    }

    Q_ASSERT(!name_.isEmpty());
    items_[name_] = this;
}

MvQFeatureGroupItem::MvQFeatureGroupItem(QString name, QString label, QString desc) :
    name_(name), label_(label), desc_(desc)
{
    if (desc_.isEmpty()) {
        desc_ = label_;
    }

    Q_ASSERT(!name_.isEmpty());
    items_[name_] = this;
}

//=================================================================
//
// MvQFeatureType
//
//=================================================================

void MvQFeatureType::load()
{
    // we only load it once
    if (items_.empty()) {
        load_system_feature_items();
        load_user_feature_items();
        load_wmo_symbols();
        config_loader();
    }
}

void MvQFeatureType::load_system_feature_items()
{
    // we only load it once
    if(!items_.empty())
        return;

    std::string symFile = metview::etcDirFile("FeatureItems");
    if (symFile.empty())
        return;

    MvRequest r;
    r.read(symFile.c_str());
    if (r) {
        do {
            const char* verb = r.getVerb();
            Q_ASSERT(verb);
            if (strcmp(verb, "item") == 0) {
                items_ << new MvQStandardFeatureType(r);
            } else if (strcmp(verb, "item_group") == 0) {
                 new MvQFeatureGroupItem(r);
            } else {
                 MvQFeatureMenuItem::add(r);
            }
        } while (r.advance());
    }
}

void MvQFeatureType::load_user_feature_items()
{
    // Get user feature items directory names
    static QList<QString> dirs = {QString::fromStdString(metview::userFeatureItemsLocalDir()),
                                  QString::fromStdString(metview::userFeatureItemsDir())};

    for (auto items_dir: dirs) {
        // Search directory to get all items (svg and png)
        QDir dir(items_dir);
        if (!dir.exists())
            break;

        dir.setNameFilters(QStringList() << "*.svg" << "*.png");
        QStringList fileList = dir.entryList();

        // Add items
        QString group = "icon";
        for (const auto& fileName : fileList) {
            QString name = fileName.left(fileName.lastIndexOf(".")).toLower();
            QString label = name;
            label.replace(0, 1, name[0].toUpper());

            if (!find(name)) {
                items_ << new MvQStandardFeatureType(name, label, fileName, items_dir, group, "marker", false);
            } else {
                std::string sTmp1 =  fileName.toStdString();
                std::string sTmp2 =  name.toStdString();
                marslog(LOG_WARN, "Cannot add user defined feature=%s. Name=%s is already in use.",
                        sTmp1.c_str(), sTmp2.c_str());
            }
        }
    }
}

void MvQFeatureType::load_wmo_symbols()
{
    WmoWeatherSymbol::init();
    for (auto const& v: WmoWeatherSymbol::elems()) {
        items_ << new MvQWmoSymbolFeatureType(v.second);
    }
}

QVector<MvQFeatureType*> MvQFeatureType::items()
{
    return items_;
}

void MvQFeatureType::config_loader()
{
    // This is how ConfigLoader initialises the registered classes
    // (e.g. MvQPalette) from the mars config.

    // Load the whole metview config. All the relevant classes will
    // load their config via ConfigLoader
    int order = 0;
    bool more = true;

    //marsinit(&argc,argv,0,0,0);
    while (more || order <= 2) {
        more = false;
        request* r = mars.setup;
        while (r) {
            if (ConfigLoader::process(r, order)) {
                more = true;
            }
            r = r->next;
        }
        order++;
    }
}

MvQFeatureType* MvQFeatureType::find(QString name)
{
    foreach (MvQFeatureType* item, items()) {
        if(item->name() == name) {
            return item;
        }
    }
    return nullptr;
}

//=================================================================
//
// MvQStandardFeatureClass
//
//=================================================================

MvQStandardFeatureType::MvQStandardFeatureType(const MvRequest& r) :
    MvQFeatureType(true)
{
    const char* verb = r.getVerb();
    Q_ASSERT(verb);
    Q_ASSERT(strcmp(verb, "item") == 0);

    if (const char* lCh = r("name")) {
        name_ = QString(lCh);
    }
    if (const char* lCh = r("label")) {
        label_ = QString(lCh);
    }
    if (const char* lCh = r("desc")) {
        desc_ = QString(lCh);
    }
    if (const char* lCh = r("icon")) {
        iconName_ = QString(lCh);
    }
    if (const char* lCh = r("type")) {
        type_ = QString(lCh);
    }
    if (const char* lCh = r("group")) {
        group_ = QString(lCh);
    }
    if (const char* lCh = r("template_colour")) {
        svgTemplateColour_ = QString(lCh);
    }

    if (desc_.isEmpty()) {
        desc_ = label_;
    }

    Q_ASSERT(!name_.isEmpty());
    Q_ASSERT(!iconName_.isEmpty());
    Q_ASSERT(!type_.isEmpty());

    //Name must be unique!

    // create pixmap
    pix_ = pixmap(defaultPixSize_, defaultPixSize_, QColor(), QPen(Qt::NoPen), QBrush(defaultPixBgColor));
}

MvQStandardFeatureType::MvQStandardFeatureType(QString name, QString label,
                                 QString iconName, QString iconDir, QString type, QString group,
                                 bool system):
    MvQFeatureType(system),
    type_(type),
    name_(name),
    label_(label),
    desc_(label),
    iconName_(iconName),
    iconDir_(iconDir),
    group_(group)
{
    Q_ASSERT(!name_.isEmpty());
    Q_ASSERT(!iconName_.isEmpty());
    Q_ASSERT(!type_.isEmpty());

    //Name must be unique!

    // create pixmap
    pix_ = pixmap(defaultPixSize_, defaultPixSize_, QColor(), QPen(defaultPixBgColor), QBrush(defaultPixBgColor));
}

QString MvQStandardFeatureType::description() const
{
    return desc_;
}

QString MvQStandardFeatureType::group() const
{
    return group_;
}

QString MvQStandardFeatureType::subGroup() const
{
    return {};
}

QPixmap MvQStandardFeatureType::pixmap(int width, int height, QColor fg, QPen /*boxPen*/, QBrush boxBrush) const
{
    auto col = ((boxBrush.style() == Qt::NoBrush)?QColor(Qt::transparent):boxBrush.color());
    if (!svgTemplateColour_.isEmpty() && fg.isValid()) {
            return MvQ::makePixmapFromSvgTemplate(iconPath(), width, height, fg, svgTemplateColour_);
    }
    return MvQ::makePixmap(iconPath(), width, height, col);
}

QString MvQStandardFeatureType::iconPath() const
{
    if (system_) {
        return QString::fromStdString(metview::systemFeatureItemsFile(iconName_.toStdString()));
    }
    else {
        return iconDir_ + "/" + iconName_;
    }
    return {};
}

//=================================================================
//
// MvQWmoSymbolFeatureClass
//
//=================================================================

MvQWmoSymbolFeatureType::MvQWmoSymbolFeatureType(WmoWeatherSymbol* symbol):
    MvQFeatureType(true),
    symbol_(symbol)
{
    pix_ = pixmap(defaultPixSize_, defaultPixSize_, QColor(Qt::black), QPen(Qt::NoPen), QBrush(Qt::NoBrush));
}

QString MvQWmoSymbolFeatureType::type() const
{
    static QString s("wmo");
    return s;
}

QString MvQWmoSymbolFeatureType::name() const
{
    if (symbol_) {
        return QString::fromStdString(symbol_->name());
    }
    return {};
}

QString MvQWmoSymbolFeatureType::label() const
{
    return name();
}

QString MvQWmoSymbolFeatureType::description() const
{
    if (symbol_) {
        return MvQ::formatBoldText(QString::fromStdString(symbol_->name())) + ":<br>" + QString::fromStdString(symbol_->title());
    }
    return {};
}

QString MvQWmoSymbolFeatureType::group() const
{
    static QString s("wmo");
    return s;
}

QString MvQWmoSymbolFeatureType::subGroup() const
{
    if (symbol_) {
        auto gr = symbol_->group();
        if (gr) {
            return QString::fromStdString(gr->name_);
        }
    }
    return {};
}

QPixmap MvQWmoSymbolFeatureType::pixmap(int width, int height, QColor fg, QPen boxPen, QBrush boxBrush) const
{
    if (symbol_) {
        return symbol_->pixmap(width, height, fg, boxPen, boxBrush);
    }
    return QPixmap(width, height);
}
