/*
    libmaus2
    Copyright (C) 2009-2015 German Tischler
    Copyright (C) 2011-2013 Genome Research Limited

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#if ! defined(TERMINATABLESYNCHRONOUSQUEUE_HPP)
#define TERMINATABLESYNCHRONOUSQUEUE_HPP

#include <libmaus2/parallel/SynchronousQueue.hpp>

#if defined(LIBMAUS2_HAVE_PTHREADS)
namespace libmaus2
{
        namespace parallel
        {
                /**
                 * posix condition variable based version
                 **/
                template<typename _value_type>
                struct TerminatableSynchronousQueue
                {
                        typedef _value_type value_type;
                        typedef TerminatableSynchronousQueue<value_type> this_type;

                        int volatile terminated;
                        libmaus2::parallel::PosixSpinLock terminatedlock;

                        int volatile qsize;
                        libmaus2::parallel::PosixSpinLock qsizelock;

                        int volatile numwaiting;
                        libmaus2::parallel::PosixSpinLock numwaitinglock;

                        pthread_mutex_t mutex;
                        pthread_cond_t cond;
                        size_t volatile numwait;
                        std::deque<value_type> Q;

                        struct MutexLock
                        {
                                pthread_mutex_t * mutex;
                                bool locked;

                                void obtain()
                                {
                                        if ( ! locked )
                                        {
                                                int const r = pthread_mutex_lock(mutex);
                                                if ( r != 0 )
                                                {
                                                        int const error = errno;
                                                        libmaus2::exception::LibMausException lme;
                                                        lme.getStream() << "MutexLock: " << strerror(error) << std::endl;
                                                        lme.finish();
                                                        throw lme;
                                                }
                                                locked = true;
                                        }
                                }

                                MutexLock(pthread_mutex_t & rmutex) : mutex(&rmutex), locked(false)
                                {
                                        obtain();
                                }

                                void release()
                                {
                                        if ( locked )
                                        {
                                                int const r = pthread_mutex_unlock(mutex);
                                                if ( r != 0 )
                                                {
                                                        int const error = errno;
                                                        libmaus2::exception::LibMausException lme;
                                                        lme.getStream() << "~MutexLock: " << strerror(error) << std::endl;
                                                        lme.finish();
                                                        throw lme;
                                                }

                                                locked = false;
                                        }
                                }

                                ~MutexLock()
                                {
                                        release();
                                }
                        };

			void initCond()
			{
				if ( pthread_cond_init(&cond,NULL) != 0 )
				{
					int const error = errno;
					libmaus2::exception::LibMausException lme;
					lme.getStream() << "PosixConditionSemaphore::initCond(): failed pthread_cond_init " << strerror(error) << std::endl;
					lme.finish();
					throw lme;
				}
			}

			void initMutex()
			{
				if ( pthread_mutex_init(&mutex,NULL) != 0 )
				{
					int const error = errno;
					libmaus2::exception::LibMausException lme;
					lme.getStream() << "PosixConditionSemaphore::initMutex(): failed pthread_mutex_init " << strerror(error) << std::endl;
					lme.finish();
					throw lme;
				}
			}


                        TerminatableSynchronousQueue()
                        : terminated(0), terminatedlock(), qsize(0), qsizelock(), numwaiting(0), numwaitinglock(), numwait(0), Q()
                        {
                        	initMutex();
                        	try
                        	{
	                        	initCond();
				}
				catch(...)
				{
					pthread_mutex_destroy(&mutex);
					throw;
				}
                        }

                        int getNumWaiting()
                        {
				int lnumwaiting;
				numwaitinglock.lock();
				lnumwaiting = numwaiting;
				numwaitinglock.unlock();
				return lnumwaiting;
                        }

                        void setTerminated()
                        {
				terminatedlock.lock();
				terminated = 1;
				terminatedlock.unlock();
                        }


                        int getTerminated()
                        {
				int lterminated;
				terminatedlock.lock();
				lterminated = terminated;
				terminatedlock.unlock();
				return lterminated;
                        }

                        void incrementQSize()
                        {
				qsizelock.lock();
				qsize += 1;
				qsizelock.unlock();
                        }

                        void decrementQSize()
                        {
				qsizelock.lock();
				qsize += 1;
				qsizelock.unlock();
                        }

                        int getQSize()
                        {
				int lqsize;
				qsizelock.lock();
				lqsize = qsize;
				qsizelock.unlock();
				return lqsize;
                        }

                        ~TerminatableSynchronousQueue()
                        {
				pthread_mutex_destroy(&mutex);
				pthread_cond_destroy(&cond);
                        }

                        void enque(value_type const v)
                        {
				if ( isTerminated() )
					throw std::runtime_error("TerminatableSynchronousQueue::enque: Queue is terminated");

                                MutexLock M(mutex);
                                Q.push_back(v);
                                pthread_cond_signal(&cond);
                                incrementQSize();
                        }

                        size_t getFillState()
                        {
				return getQSize();
                        }

                        bool isTerminated()
                        {
				return getTerminated();
			}

                        void terminate()
                        {
				setTerminated();

				while ( getNumWaiting() > 0 )
				{
					MutexLock M(mutex);
					pthread_cond_signal(&cond);
				}
                        }

                        struct NumWaitingObject
                        {
				int volatile & numwaiting;
				libmaus2::parallel::PosixSpinLock & numwaitinglock;

				NumWaitingObject(
					int volatile & rnumwaiting,
					libmaus2::parallel::PosixSpinLock & rnumwaitinglock
				) : numwaiting(rnumwaiting), numwaitinglock(rnumwaitinglock)
				{
					libmaus2::parallel::ScopePosixSpinLock slock(numwaitinglock);
					numwaiting += 1;
				}

				~NumWaitingObject()
				{
					libmaus2::parallel::ScopePosixSpinLock slock(numwaitinglock);
					numwaiting -= 1;
				}
			};

                        value_type deque()
                        {
				NumWaitingObject NWO(numwaiting,numwaitinglock);

                                MutexLock M(mutex);

                                while ( true )
                                {
					if ( getQSize() )
					{
						value_type v = Q.front();
						Q.pop_front();
						decrementQSize();
						return v;
					}
					else if ( isTerminated() )
					{
						throw std::runtime_error("TerminatableSynchronousQueue::deque: Queue is terminated");
					}
					else
					{
						int const r = pthread_cond_wait(&cond,&mutex);
						if ( r != 0 )
						{
							int const error = errno;
							libmaus2::exception::LibMausException lme;
							lme.getStream() << "TerminatableSynchronousQueue::deque: pthread_cond_wait " << strerror(error) << std::endl;
							lme.finish();
							throw lme;
						}
					}
                                }
                        }
                };
        }
}
#endif
#endif
