/*****************************************************************************
 * Copyright (C) 2016 Krusader Krew                                          *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This package is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this package; if not, write to the Free Software               *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA *
 *****************************************************************************/

#include "krjob.h"

#include <KI18n/KLocalizedString>
#include <KIO/DeleteJob>
#include <KIO/FileUndoManager>

KrJob *KrJob::createCopyJob(KIO::CopyJob::CopyMode mode, const QList<QUrl> &src,
                            const QUrl &destination, KIO::JobFlags flags)
{
    Type type;
    QString description;
    switch (mode) {
    case KIO::CopyJob::Copy:
        type = Copy;
        description = i18n("Copy to %1", destination.toDisplayString());
        break;
    case KIO::CopyJob::Move:
        type = Move;
        description = i18n("Move to %1", destination.toDisplayString());
        break;
    case KIO::CopyJob::Link:
        type = Link;
        description = i18n("Link to %1", destination.toDisplayString());
        break;
    }

    return new KrJob(type, src, destination, flags, description);
}

KrJob *KrJob::createDeleteJob(const QList<QUrl> &urls, bool moveToTrash)
{
    const Type type = moveToTrash ? Trash : Delete;
    const bool oneFile = urls.length() == 1;
    const QString description =
        moveToTrash ? (oneFile ? i18n("Move %1 to trash", urls.first().toDisplayString()) :
                                 i18np("Move %1 file to trash", "Move %1 files to trash", urls.length())) :
                      (oneFile ? i18n("Delete %1", urls.first().toDisplayString()) :
                                 i18np("Delete %1 file", "Delete %1 files", urls.length()));

    return new KrJob(type, urls, QUrl(), KIO::DefaultFlags, description);
}

KrJob::KrJob(Type type, const QList<QUrl> &urls, const QUrl &dest, KIO::JobFlags flags,
             const QString &description)
    : m_type(type), m_urls(urls), m_dest(dest), m_flags(flags), m_description(description), m_job(0)
{
}

void KrJob::start()
{
    if (m_job) {
        // job was already started
        m_job->resume();
        return;
    }

    switch (m_type) {
    case Copy: {
        KIO::CopyJob *job = KIO::copy(m_urls, m_dest, m_flags);
        KIO::FileUndoManager::self()->recordCopyJob(job);
        m_job = job;
        break;
    }
    case Move: {
        KIO::CopyJob *job = KIO::move(m_urls, m_dest, m_flags);
        KIO::FileUndoManager::self()->recordCopyJob(job);
        m_job = job;
        break;
    }
    case Link: {
        KIO::CopyJob *job = KIO::link(m_urls, m_dest, m_flags);
        KIO::FileUndoManager::self()->recordCopyJob(job);
        m_job = job;
        break;
    }
    case Trash: {
        m_job = KIO::trash(m_urls);
        KIO::FileUndoManager::self()->recordJob(KIO::FileUndoManager::Trash, m_urls, QUrl("trash:/"),
                                                m_job);
        break;
    }
    case Delete:
        m_job = KIO::del(m_urls);
    }

    connect(m_job, &KIO::Job::finished, this, [=]() {
        emit terminated(this);
        deleteLater();
    });

    emit started(m_job);
}

void KrJob::cancel()
{
    if (m_job)
        m_job->kill();
    else {
        emit terminated(this);
        deleteLater();
    }
}

void KrJob::pause()
{
    if (m_job)
        m_job->suspend();
}
