/*
 * Copyright (C) 2014 Guitarix project MOD project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */


////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_common.h"      // faust support and denormal protection (SSE)
#include "gx_bypass.cc"
#include "gx_bossds1.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "gx_resampler.h"
#include "bossds1.cc"    // dsp class generated by faust -> dsp2cc

////////////////////////////// PLUG-IN CLASS ///////////////////////////

namespace bossds1 {

class Gx_bossds1_
{
private:
  GxBypass                     bp;
  float*                       bypass;
  DenormalProtection           MXCSR;
  // pointer to buffer
  float*      output;
  float*      input;
  // pointer to dsp class
  PluginLV2*  bossds1;
  // private functions
  inline void run_dsp_(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);
  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  Gx_bossds1_();
  ~Gx_bossds1_();
};

// constructor
Gx_bossds1_::Gx_bossds1_() :
  bp(),
  bypass(0),
  MXCSR(),
  output(NULL),
  input(NULL),
  bossds1(bossds1::plugin()) {};

// destructor
Gx_bossds1_::~Gx_bossds1_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (bossds1->activate_plugin !=0)
    bossds1->activate_plugin(false, bossds1);
  // delete DSP class
  bossds1->delete_instance(bossds1);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_bossds1_::init_dsp_(uint32_t rate)
{
  bp.init_bypass(rate);
  bossds1->set_samplerate(rate, bossds1); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_bossds1_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
    case BYPASS: 
      bypass = static_cast<float*>(data); // , 0.0, 0.0, 1.0, 1.0 
      break;
    default:
      break;
    }
}

void Gx_bossds1_::activate_f()
{
  // allocate the internal DSP mem
  if (bossds1->activate_plugin !=0)
    bossds1->activate_plugin(true, bossds1);
}

void Gx_bossds1_::clean_up()
{
  // delete the internal DSP mem
  if (bossds1->activate_plugin !=0)
    bossds1->activate_plugin(false, bossds1);
}

void Gx_bossds1_::deactivate_f()
{
  // delete the internal DSP mem
  if (bossds1->activate_plugin !=0)
    bossds1->activate_plugin(false, bossds1);
}

void Gx_bossds1_::run_dsp_(uint32_t n_samples)
{
  if (n_samples< 1) return;
  MXCSR.set_();
  // run dsp
  FAUSTFLOAT buf[n_samples];
  // do inplace processing at default
  if (output != input)
    memcpy(output, input, n_samples*sizeof(float));
  // check if bypass is pressed
  if (!bp.pre_check_bypass(bypass, buf, input, n_samples)) 
    bossds1->mono_audio(static_cast<int>(n_samples), input, output, bossds1);
  bp.post_check_bypass(buf, output, n_samples);

  MXCSR.reset_();
}

void Gx_bossds1_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  bossds1->connect_ports(port,  data, bossds1);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Handle 
Gx_bossds1_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_bossds1_ *self = new Gx_bossds1_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_((uint32_t)rate);

  return (LV2_Handle)self;
}

void Gx_bossds1_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_bossds1_*>(instance)->connect_all__ports(port, data);
}

void Gx_bossds1_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_bossds1_*>(instance)->activate_f();
}

void Gx_bossds1_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  static_cast<Gx_bossds1_*>(instance)->run_dsp_(n_samples);
}

void Gx_bossds1_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_bossds1_*>(instance)->deactivate_f();
}

void Gx_bossds1_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_bossds1_* self = static_cast<Gx_bossds1_*>(instance);
  self->clean_up();
  delete self;
}

const LV2_Descriptor Gx_bossds1_::descriptor =
{
  GXPLUGIN_URI "#_bossds1_",
  Gx_bossds1_::instantiate,
  Gx_bossds1_::connect_port,
  Gx_bossds1_::activate,
  Gx_bossds1_::run,
  Gx_bossds1_::deactivate,
  Gx_bossds1_::cleanup,
  NULL
};


} // end namespace bossds1

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

extern "C"
LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &bossds1::Gx_bossds1_::descriptor;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
