#include "Sim/Fitting/ObjectiveMetric.h"
#include "Sim/Fitting/ObjectiveMetricUtil.h"
#include "Tests/GTestWrapper/google_test.h"
#include <cmath>

TEST(ObjectiveMetricTest, Chi2WellFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 3.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 4.0, 3.0};
    std::vector<double> uncertainties{0.1, 0.1, 0.5, 0.5};
    std::vector<double> weight_factors{1.0, 1.0, 1.0, 2.0};

    Chi2Metric metric;

    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, sim_data, uncertainties, weight_factors),
                     0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, sim_data, weight_factors), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors),
                     212.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 5.0);

    std::vector<double> exp_data_1 = exp_data;
    exp_data_1[0] = -1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data_1, uncertainties, weight_factors),
                     112.0);

    std::vector<double> uncertainties_1 = uncertainties;
    uncertainties_1[3] = -1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties_1, weight_factors),
                     204.0);
    uncertainties_1[3] = 0.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties_1, weight_factors),
                     204.0);

    metric.setNorm(ObjectiveMetricUtil::l1Norm());
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors),
                     26.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 5.0);

    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}, {}), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}), 0.0);
}

TEST(ObjectiveMetricTest, Chi2IllFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 3.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 4.0, 3.0};
    std::vector<double> uncertainties{0.1, 0.1, 0.5, 0.5};
    std::vector<double> weight_factors{1.0, 1.0, 1.0, 2.0};

    Chi2Metric metric;

    EXPECT_THROW(metric.computeFromArrays({}, {}, {}, {1.0}), std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays({}, {}, {1.0}), std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays(sim_data, exp_data, {}, weight_factors),
                 std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays({}, exp_data, weight_factors), std::runtime_error);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = -1.0;
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, uncertainties, weight_factors),
                 std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, weight_factors),
                 std::runtime_error);

    std::vector<double> weight_factors_1(weight_factors.size(), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors_1),
                     0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors_1), 0.0);

    std::vector<double> uncertainties_1 = uncertainties;
    uncertainties_1[0] = std::numeric_limits<double>::min();
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties_1, weight_factors),
                     std::numeric_limits<double>::max());
}

TEST(ObjectiveMetricTest, PoissionLikeWellFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 4.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 5.0, 3.0};
    std::vector<double> weight_factors{1.0, 1.0, 1.0, 2.0};

    PoissonLikeMetric metric;

    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, sim_data, weight_factors), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 2.25);

    std::vector<double> exp_data_1 = exp_data;
    exp_data_1[0] = -1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data_1, weight_factors), 1.25);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = 0.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data_1, exp_data, weight_factors), 5.25);

    metric.setNorm(ObjectiveMetricUtil::l1Norm());

    std::vector<double> sim_data_2 = sim_data;
    sim_data_2[1] = 1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data_2, exp_data, weight_factors), 2.5);

    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}, {}), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}), 0.0);
}

TEST(ObjectiveMetricTest, PoissionLikeIllFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 3.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 4.0, 3.0};
    std::vector<double> weight_factors{1.0, 1.0, 1.0, 2.0};

    PoissonLikeMetric metric;

    EXPECT_THROW(metric.computeFromArrays({}, {}, {}, {1.0}), std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays({}, {}, {1.0}), std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays(sim_data, exp_data, {}, weight_factors),
                 std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays({}, exp_data, weight_factors), std::runtime_error);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = -1.0;
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, weight_factors),
                 std::runtime_error);

    std::vector<double> weight_factors_1(weight_factors.size(), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors_1), 0.0);
}

TEST(ObjectiveMetricTest, LogWellFormed)
{
    std::vector<double> sim_data{1.0, 10.0, 1e2, 1e4};
    std::vector<double> exp_data{10.0, 1.0, 1e3, 1e5};
    std::vector<double> uncertainties{0.1, 0.1, 0.5, 0.5};
    std::vector<double> weight_factors{1.0, 1.0, 1.0, 2.0};

    LogMetric metric;

    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, sim_data, uncertainties, weight_factors),
                     0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, sim_data, weight_factors), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors),
                     8.00040101e10 * std::log(10.0) * std::log(10.0));
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 5.0);

    std::vector<double> exp_data_1 = exp_data;
    exp_data_1[0] = -1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data_1, uncertainties, weight_factors),
                     8.00040001e10 * std::log(10.0) * std::log(10.0));
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data_1, weight_factors), 4.0);

    std::vector<double> uncertainties_1 = uncertainties;
    uncertainties_1[3] = -1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties_1, weight_factors),
                     4.0101e6 * std::log(10.0) * std::log(10.0));
    uncertainties_1[3] = 0.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties_1, weight_factors),
                     4.0101e6 * std::log(10.0) * std::log(10.0));

    metric.setNorm(ObjectiveMetricUtil::l1Norm());
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors),
                     4.0211e5 * std::log(10.0));
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 5.0);

    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}, {}), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}), 0.0);
}

TEST(ObjectiveMetricTest, LogIllFormed)
{
    std::vector<double> sim_data{1.0, 10.0, 1e2, 1e4};
    std::vector<double> exp_data{10.0, 1.0, 1e3, 1e5};
    std::vector<double> uncertainties{0.1, 0.1, 0.5, 0.5};
    std::vector<double> weight_factors{1.0, 1.0, 1.0, 2.0};

    LogMetric metric;

    EXPECT_THROW(metric.computeFromArrays({}, {}, {}, {1.0}), std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays({}, {}, {1.0}), std::runtime_error);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = -1.0;
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, uncertainties, weight_factors),
                 std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, weight_factors),
                 std::runtime_error);

    std::vector<double> weight_factors_1(weight_factors.size(), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors_1),
                     0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors_1), 0.0);

    std::vector<double> uncertainties_1 = uncertainties;
    uncertainties_1[0] = std::numeric_limits<double>::min();
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties_1, weight_factors),
                     std::numeric_limits<double>::max());
}

TEST(ObjectiveMetricTest, meanRelativeDifferenceWellFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 4.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 2.0, 2.0};
    std::vector<double> uncertainties{1.0, 1.0, 2.0, 1.0};
    std::vector<double> weight_factors{1.0, 1.0, 2.0, 1.0};

    meanRelativeDifferenceMetric metric;

    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, sim_data, weight_factors), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 5.0 / 9.0);

    std::vector<double> exp_data_1 = exp_data;
    exp_data_1[0] = -1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data_1, weight_factors), 4.0 / 9.0);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = 0.0;
    exp_data_1[0] = 0.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data_1, exp_data_1, weight_factors), 4.0 / 9.0);

    metric.setNorm(ObjectiveMetricUtil::l1Norm());
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 5.0 / 3.0);

    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors),
                     6.0);

    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}, {}), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}), 0.0);
}

TEST(ObjectiveMetricTest, meanRelativeDifferenceIllFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 4.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 3.0, 2.0};
    std::vector<double> uncertainties{1.0, 1.0, 2.0, 1.0};
    std::vector<double> weight_factors{1.0, 1.0, 2.0, 1.0};

    meanRelativeDifferenceMetric metric;

    EXPECT_THROW(metric.computeFromArrays(sim_data, {}, uncertainties, weight_factors),
                 std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays(sim_data, {}, weight_factors), std::runtime_error);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = -1.0;
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, uncertainties, weight_factors),
                 std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, weight_factors),
                 std::runtime_error);

    std::vector<double> weight_factors_1(weight_factors.size(), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors_1),
                     0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors_1), 0.0);
}

TEST(ObjectiveMetricTest, RQ4WellFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 4.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 2.0, 2.0};
    std::vector<double> uncertainties{1.0, 1.0, 2.0, 1.0};
    std::vector<double> weight_factors{1.0, 1.0, 2.0, 1.0};

    RQ4Metric metric;

    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, sim_data, weight_factors), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 14.0);

    std::vector<double> exp_data_1 = exp_data;
    exp_data_1[0] = -1.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data_1, weight_factors), 13.0);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = 0.0;
    exp_data_1[0] = 0.0;
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data_1, exp_data_1, weight_factors), 13.0);

    metric.setNorm(ObjectiveMetricUtil::l1Norm());
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors), 8.0);

    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors),
                     6.0);

    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}, {}), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays({}, {}, {}), 0.0);
}

TEST(ObjectiveMetricTest, RQ4IllFormed)
{
    std::vector<double> sim_data{1.0, 2.0, 4.0, 4.0};
    std::vector<double> exp_data{2.0, 1.0, 3.0, 2.0};
    std::vector<double> uncertainties{1.0, 1.0, 2.0, 1.0};
    std::vector<double> weight_factors{1.0, 1.0, 2.0, 1.0};

    RQ4Metric metric;

    EXPECT_THROW(metric.computeFromArrays(sim_data, exp_data, {}, weight_factors),
                 std::runtime_error);
    EXPECT_THROW(metric.computeFromArrays({}, exp_data, weight_factors), std::runtime_error);

    std::vector<double> sim_data_1 = sim_data;
    sim_data_1[0] = -1.0;
    EXPECT_THROW(metric.computeFromArrays(sim_data_1, exp_data, uncertainties, weight_factors),
                 std::runtime_error);

    std::vector<double> weight_factors_1(weight_factors.size(), 0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, uncertainties, weight_factors_1),
                     0.0);
    EXPECT_DOUBLE_EQ(metric.computeFromArrays(sim_data, exp_data, weight_factors_1), 0.0);
}

TEST(ObjectiveMetricTest, createMetric)
{
    auto result = ObjectiveMetricUtil::createMetric("Poisson-like");
    EXPECT_TRUE(dynamic_cast<PoissonLikeMetric*>(result.get()));
    // Since norm functions lack equality comparison, check the equality of applying them
    EXPECT_DOUBLE_EQ(result->norm()(2.0), ObjectiveMetricUtil::l2Norm()(2.0));

    result = ObjectiveMetricUtil::createMetric("Poisson-Like", "L1");
    EXPECT_TRUE(dynamic_cast<PoissonLikeMetric*>(result.get()));
    EXPECT_DOUBLE_EQ(result->norm()(2.0), ObjectiveMetricUtil::l1Norm()(2.0));

    result = ObjectiveMetricUtil::createMetric("poisson-like", "l1");
    EXPECT_TRUE(dynamic_cast<PoissonLikeMetric*>(result.get()));
    EXPECT_DOUBLE_EQ(result->norm()(2.0), ObjectiveMetricUtil::l1Norm()(2.0));

    result = ObjectiveMetricUtil::createMetric("poissoN-likE", "L2");
    EXPECT_TRUE(dynamic_cast<PoissonLikeMetric*>(result.get()));
    EXPECT_DOUBLE_EQ(result->norm()(2.0), ObjectiveMetricUtil::l2Norm()(2.0));

    result = ObjectiveMetricUtil::createMetric("poisson-like");
    EXPECT_TRUE(dynamic_cast<PoissonLikeMetric*>(result.get()));
    EXPECT_DOUBLE_EQ(result->norm()(2.0), ObjectiveMetricUtil::l2Norm()(2.0));

    result = ObjectiveMetricUtil::createMetric("chi2");
    EXPECT_TRUE(dynamic_cast<Chi2Metric*>(result.get()));

    result = ObjectiveMetricUtil::createMetric("Chi2");
    EXPECT_TRUE(dynamic_cast<Chi2Metric*>(result.get()));

    result = ObjectiveMetricUtil::createMetric("Log");
    EXPECT_TRUE(dynamic_cast<LogMetric*>(result.get()));

    result = ObjectiveMetricUtil::createMetric("log");
    EXPECT_TRUE(dynamic_cast<LogMetric*>(result.get()));

    result = ObjectiveMetricUtil::createMetric("reldiff");
    EXPECT_TRUE(dynamic_cast<meanRelativeDifferenceMetric*>(result.get()));

    result = ObjectiveMetricUtil::createMetric("RelDiff");
    EXPECT_TRUE(dynamic_cast<meanRelativeDifferenceMetric*>(result.get()));

    result = ObjectiveMetricUtil::createMetric("RQ4");
    EXPECT_TRUE(dynamic_cast<RQ4Metric*>(result.get()));

    result = ObjectiveMetricUtil::createMetric("rq4");
    EXPECT_TRUE(dynamic_cast<RQ4Metric*>(result.get()));
}
