/* SPDX-FileCopyrightText: 2023 Blender Authors
 *
 * SPDX-License-Identifier: GPL-2.0-or-later */

/**
 * This pass scans the tile mask generated by the classify step and output indirect dispatch args.
 *
 * Dispatched as one thread for each trace resolution tile.
 */

#pragma BLENDER_REQUIRE(gpu_shader_utildefines_lib.glsl)
#pragma BLENDER_REQUIRE(gpu_shader_math_vector_lib.glsl)
#pragma BLENDER_REQUIRE(gpu_shader_codegen_lib.glsl)
#pragma BLENDER_REQUIRE(eevee_gbuffer_lib.glsl)

void main()
{
  ivec2 tile = ivec2(gl_GlobalInvocationID.xy);

  bool tracing_tile_is_used = false;
  for (int x = 0; x < uniform_buf.raytrace.resolution_scale; x++) {
    for (int y = 0; y < uniform_buf.raytrace.resolution_scale; y++) {
      ivec2 full_res_tile = tile * uniform_buf.raytrace.resolution_scale + ivec2(x, y);
      if (any(greaterThanEqual(full_res_tile, imageSize(tile_mask_img)))) {
        continue;
      }
      bool denoise_tile_is_used = imageLoad(tile_mask_img, full_res_tile).r != 0u;
      if (denoise_tile_is_used) {
        /* Dispatch full resolution denoise tile. */
        uint tile_index = atomicAdd(denoise_dispatch_buf.num_groups_x, 1u);
        denoise_tiles_buf[tile_index] = packUvec2x16(uvec2(full_res_tile));

        tracing_tile_is_used = true;
      }
    }
  }

  if (tracing_tile_is_used) {
    /* Dispatch trace resolution tracing tile. */
    uint tile_index = atomicAdd(ray_dispatch_buf.num_groups_x, 1u);
    ray_tiles_buf[tile_index] = packUvec2x16(uvec2(tile));
  }
}
