/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Remez poly at point for Y1

Generated by SageMath and Sollya:
```python
def compute_intervals(zeros):
    intervals = []
    for i in range(0, len(zeros)):
        if i == 0:
            a = 1.4 - zeros[i]
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.03 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        elif i + 1 > len(zeros) - 1:
            a = (zeros[i - 1] + zeros[i]) / 2 - 0.03 - zeros[i]
            b = (zeros[i]) + 0.83 + 0.03 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        else:
            a = (zeros[i - 1] + zeros[i]) / 2 - zeros[i] - 0.03
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.03  - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
    return intervals

intervals = compute_intervals(y1_zeros)
print(intervals)

def build_sollya_script(a, b, zero, deg):
    return f"""
prec = 200;
bessel_y1 = library("/Users/radzivon/RustroverProjects/pxfm/notes/bessel_sollya/cmake-build-release/libbessel_sollya.dylib");
f = bessel_y1(x + {zero});
d = [{a}, {b}];
pf = remez(f, {deg}, d);
for i from 0 to degree(pf) do {{
    write(coeff(pf, i)) >> "coefficients.txt";
    write("\\n") >> "coefficients.txt";
}};
"""

def load_coefficients(filename):
    with open(filename, "r") as f:
        return [RealField(500)(line.strip()) for line in f if line.strip()]

def call_sollya_on_interval(a, b, zero, degree=12):
    sollya_script = build_sollya_script(a, b, zero, degree)
    with open("tmp_interval.sollya", "w") as f:
        f.write(sollya_script)
    import subprocess
    if os.path.exists("coefficients.txt"):
        os.remove("coefficients.txt")
    try:
        result = subprocess.run(
            ["sollya", "tmp_interval.sollya"],
            check=True,
            capture_output=True,
            text=True
        )
    except subprocess.CalledProcessError as e:
        return

def print_remez_coeffs(poly):
    print("[")
    for i in range(len(poly)):
        coeff = poly[i]
        print_double_double("", coeff)
    print("],")

degree = 27

print(f"pub(crate) static Y1_COEFFS: [[(u64, u64); {degree + 1}]; {len(intervals)}] = [")
for i in range(0, len(intervals)):
    interval = intervals[i]
    call_sollya_on_interval(interval[0], interval[1], interval[2], degree)
    coeffs = load_coefficients(f"coefficients.txt")
    print_remez_coeffs(coeffs)
print("];")
```
**/
pub(crate) static Y1_COEFFS_REMEZ: [[(u64, u64); 28]; 32] = [
    [
        (0xb839eddec375bf6c, 0x3b9d10948a169574),
        (0xbc74a40e665752d5, 0x3fe0aa48442f014b),
        (0xbc5d63ac05d1c2a6, 0xbfbe56f82217b8f1),
        (0x3c1a86d2f484de78, 0xbfa0d2af4e932384),
        (0xbc18399f4b2b50aa, 0xbf73a6dec3729034),
        (0x3c0746a596f9ad9c, 0x3f7e671c7d1193a4),
        (0x3bf3d92bfdba1d2f, 0xbf65429dc5a47c3d),
        (0xbbf4b8971a16b469, 0x3f517ab4afad8d4c),
        (0x3b652513f456a41f, 0xbf40b2d877c8c1c4),
        (0xbbcc3336d312aa0f, 0x3f2eea7bbac80106),
        (0xbbaf716735bf35cc, 0xbf1c3fae403eeb85),
        (0xbbad1a66b3876cc3, 0x3f09d174ea03d698),
        (0x3b8d9b871bbf32f3, 0xbef7956a25d2646a),
        (0xbb773ed35ee47138, 0x3ee58659a1776ea8),
        (0xbb62769d658c2265, 0xbed3a28e8e3dbde3),
        (0xbb552085e0470bd4, 0x3ec1e78375f4f94a),
        (0xbb08d21567003010, 0xbeb050ddfe36a8d4),
        (0x3ad34a82d3857504, 0x3e9db7fda6f15f12),
        (0x3b2eab75515a5387, 0xbe8b2f16073b5272),
        (0x3b154bf5b63c1a3c, 0x3e78ee8a3e626df9),
        (0xbae94fc562d9a8af, 0xbe65dbfe34a22d4b),
        (0xbafe6d5418c52ec0, 0x3e52ed2e77b64c13),
        (0xbae2e47cbd9517fb, 0xbe45f153b0798574),
        (0x3ad48498c8542ad5, 0x3e37beb356ba6efd),
        (0xba9b754aa573c033, 0xbe026b71e3609cb5),
        (0xba952f344bc0c1b0, 0xbdf0ee25335de707),
        (0xba9d8998731da058, 0xbe10333960f701ed),
        (0x3aa4fd188b620a2f, 0x3e02f1394b2fc331),
    ],
    [
        (0x3c3c696f3e8f595c, 0x3fdaabb4011ed330),
        (0x37349e0b66c92237, 0x3aa5d4664eb6a104),
        (0xbc682b7fdbcd42e2, 0xbfc8b45babe797ba),
        (0x3bf25562940422c5, 0x3f8e147099a6f0ea),
        (0xbbfea4fe1c2176f7, 0x3f88c5af1eeb2d6b),
        (0xbbc92561b6f1afd0, 0xbf4133fa47da52e2),
        (0xbbd2318dbab67c78, 0xbf3bf8af93ff0b8c),
        (0x3b8367aefe821e18, 0x3f021d64be62ccf7),
        (0xbb5ba2bd39c31481, 0x3eb44d2ce67b2ce0),
        (0x3b33f47634278a41, 0x3eb14c3ab2ad7c72),
        (0xbb35600f33967647, 0xbe9b8eee52e1c5c0),
        (0x3b1e72487b134917, 0x3e7a85f1a876eaac),
        (0x3af57616cb55d15e, 0xbe5bf86b3c8c3956),
        (0xbacc84ceaea79fcd, 0x3e3f01920ab67df5),
        (0xbacf7e9704007b68, 0xbe20fac5449162d1),
        (0xba8a696b60bf9429, 0x3e0281882dcdc5b5),
        (0xba74ae0431042976, 0xbde42b5ce9ff90d4),
        (0x3a4ccb47728927b2, 0x3dc5f94ad54c0b9a),
        (0x3a2998fdce097e66, 0xbda7ed4bc62e5bfe),
        (0x3a25b8d3d29e6d16, 0x3d8a0a9970b12e30),
        (0xb9fd16bf93867f52, 0xbd6c55bc3e323959),
        (0xb9be0d077b14dc04, 0x3d4eddb2bc728a6f),
        (0x39bef3623a05f15f, 0xbd30ceb720a28712),
        (0xb9af97a81acfc768, 0x3d11f5e51ac3427a),
        (0x3990a840895512d8, 0xbcf30a29d26fe9e8),
        (0xb97f8b7789198fa8, 0x3cd774949aecc05b),
        (0x3930f32d60ee9695, 0xbcbee41d171d95d6),
        (0xb92bb8694cebdcf4, 0x3c96d81ef4f30d1d),
    ],
    [
        (0xb5bd273adffce0f7, 0xb95d4023dff47874),
        (0x3c7b8d2a1c4966d1, 0xbfd5c7c556f0c19a),
        (0xbc370c18fffaa71b, 0x3fa00b9f8571ca1f),
        (0xbc4e4289c3a5bb03, 0x3faa15d92dfe3e27),
        (0xbbf5a35920e4110a, 0xbf710a329e2c23fa),
        (0xbc05118854f257ec, 0xbf61be6db9923acf),
        (0x3bcb875e5c6cb75b, 0x3f2337c7e138f484),
        (0x3ba72a340f9a98f1, 0x3f085b940eb607f9),
        (0xbb21a52945306bc3, 0xbec80619147b78f2),
        (0xbb469f0f1be4d82d, 0xbea255e6cf70cf33),
        (0x3ae6cf00d3f14236, 0x3e5b62cd0201498a),
        (0x3ad09fb089acb5d5, 0x3e380b1aac007d45),
        (0x3a846c00973acf35, 0xbdfa7ee05a569284),
        (0xba2f88c51e1b4b4a, 0x3d84065ca5ea7a77),
        (0x3a12f4d97219bd18, 0xbd93847684c52488),
        (0xba07e0dcba44e25e, 0x3d74980895f648d3),
        (0x39efb8f4f34b685f, 0xbd4c00f7d7a2505b),
        (0xb9b0c1aa67b8ed6e, 0x3d2402fd40ed4b9e),
        (0x3993d56fd395d2c5, 0xbcfded828c8c4b93),
        (0xb95fd48828b69170, 0x3cd6381399866139),
        (0xb953bf2573b4d2c9, 0xbcb06b3a2379b696),
        (0xb92dc38af391a21b, 0x3c884327396947d2),
        (0xb90f7aa26b6f4550, 0xbc61ee8868358f55),
        (0xb8db41373ba1af43, 0x3c3a9d108cee5321),
        (0x38b6c01210cd7020, 0xbc1373d3a1131d7c),
        (0x387b749adc7b48d6, 0x3bea94db4c089199),
        (0xb86a113c4f1a075b, 0xbbc7ddb04ee5e8fa),
        (0x38478b50ffaa017a, 0x3ba9bccbc209b225),
    ],
    [
        (0xbc583acf6511df31, 0xbfd36732d4b96094),
        (0x35bb32394e3544b3, 0xb9504d29c8d39de2),
        (0xbc5b435291c33435, 0x3fc3001c8002caf8),
        (0x3c085153ba2c8a0d, 0xbf7bf5a03bab4999),
        (0x3c28d66c54022151, 0xbf8751ea028c1953),
        (0x3be6d2f42909876b, 0x3f423874cd8d0401),
        (0xbbb4d7b925e5e573, 0x3f364f6610d64939),
        (0xbb9920eb8c3a22f7, 0xbef02978de3838b9),
        (0x3b66c8e6bd84e120, 0xbed72f0766d0d591),
        (0x3b20b7ba8289b800, 0x3e8f2081874c7e33),
        (0xbafe9312e3e2eb2a, 0x3e6defd5dce00666),
        (0x3ab45c8d50dfdf07, 0xbe2205c6fe8f26fd),
        (0x3a63094c526b5e53, 0xbdfb6432e005a435),
        (0x3a5f030235b03180, 0x3db0289fce476893),
        (0x3a06336d9bbe36cd, 0x3d80778a2d37fd6e),
        (0x39ca74bc4d2363cc, 0xbd29723fe568d2cc),
        (0xb9a9854b8c15ca93, 0xbd082f3f7062fec4),
        (0xb960225f10f6c2a9, 0x3cc7a09f41509f4d),
        (0x38e482bb3a554a6f, 0xbc85319ecc467384),
        (0x390dda9bdc9a0a0c, 0x3c62d63451822c2b),
        (0xb8c73bb6bf497904, 0xbc38c3eac5e357dd),
        (0xb8814c483642d1a7, 0x3c0ba16f9ef7d844),
        (0x3871b7e296e61522, 0xbbdf92d4713ff54f),
        (0xb84e39cdf13f5f7c, 0x3bb25448a2561236),
        (0x37fdcd7885723648, 0xbb85269652dc40ed),
        (0x37e69c4b0a60ddd9, 0x3b58bb2c31aabdf7),
        (0x37cd1cbdeb5d68ef, 0xbb2f0558fc9fc6be),
        (0xb77691ca0957001c, 0x3aff85ebcabb7e09),
    ],
    [
        (0xb5d0a191002f7d89, 0x3946a82e829b73d0),
        (0x3c61dc672a53c58f, 0x3fd15f993fceab5c),
        (0xbc1455aa7db14f8f, 0xbf902b3933cf21b1),
        (0x3c443ede64b782da, 0xbfa6395dfe49fcd4),
        (0x3be10ad2b71b989e, 0x3f63ced2a2e69180),
        (0x3c0f30cad7e0eac9, 0x3f607a678d6000bb),
        (0x3baf8fd02659436b, 0xbf1b50d7e1d32020),
        (0x3b91fb17828700ad, 0xbf06f7bab104f355),
        (0xbb60ab92716b77c6, 0x3ec176e72bf94b8f),
        (0x3b31557f37d0bc88, 0x3ea2becb2b6bbc6d),
        (0x3aea30df2d48cbea, 0xbe5a384eebfe1367),
        (0xbad27cd4c55aaf05, 0xbe341e7a923c3a90),
        (0x3a8bfdf2814022d5, 0x3de9e3284f5dd101),
        (0xba5d8b8089251a1e, 0x3dbec40b623ea350),
        (0xba1f7c34ccfe84c2, 0xbd726868b83f8d7d),
        (0x39eff9c315ea063c, 0xbd416f7ae80cad5e),
        (0x3901c0bf665fc134, 0x3cf318658ecb9f66),
        (0xb95a3e748f544b78, 0x3cbfacabdbab4e26),
        (0xb9045493a1f3354c, 0xbc713cc1f69cc0fc),
        (0xb8b0df6f123cf21e, 0xbc33cb940319e5c9),
        (0x3863c569980a714d, 0x3bda2202db83e3df),
        (0xb85ff8178cbd6f47, 0x3bb47d99d87a6c81),
        (0x381b27b5c35fabff, 0xbb7452b1dfb57b43),
        (0x37dfa437b73973d9, 0x3b350569c6b6e6ea),
        (0xb7ade85c8a408aba, 0xbb080cb19b69b6d0),
        (0xb7663e78d5cbec8e, 0x3ad82b8c2a256a02),
        (0xb732862a69eab24c, 0xbaa7b1cc7eec05b9),
        (0x371988acfb374395, 0x3a78a467c1568a16),
    ],
    [
        (0xbc64f19a2762f5ae, 0x3fd00ef3745e0e3c),
        (0xb5f8d8215e0ae323, 0x39749943012b8dfc),
        (0xbc2a5ca3437ac1e3, 0xbfbfcdacdda138f2),
        (0x3c19a8e604f9385e, 0x3f706cc34cd829fa),
        (0x3c01d3813b5701b6, 0x3f84641bb10c16cb),
        (0x3bbd08ac252b4bd1, 0xbf37fac943e2a16d),
        (0xbbd98c55ed55e628, 0xbf34769ed32e14a2),
        (0xbb8b665d8b88d4a7, 0x3ee80608ecda1520),
        (0xbb7d207258853e42, 0x3ed5cc8242d77e79),
        (0x3b2f5c5f13a6d47c, 0xbe888c8f253923cc),
        (0xbb0d50fa940eb605, 0xbe6ce5908c1f1376),
        (0x3abc197e72c6469e, 0x3e1ed1625825eeeb),
        (0x3a969b2f19a7ed88, 0x3dfa30d624f10edf),
        (0x3a41771a6391a818, 0xbdaa50765ef82dbe),
        (0x39eb336c0c56cc4b, 0xbd814cd364ef8287),
        (0xb9d42fa2e0b554e2, 0x3d306199db2399a2),
        (0x39a86abea2a826ff, 0x3d0164150f087364),
        (0xb910d2f6ebf5241d, 0xbcaefb1085d3b487),
        (0x390479f708f87e53, 0xbc7ba1419f5a8512),
        (0xb8cf5a4e3ac59e65, 0x3c27638ccf278b3d),
        (0xb89307e1a0fbb15c, 0x3bf18a62dcf6ad4f),
        (0x3829bd1dae700b20, 0xbb9b54a49cff984a),
        (0x37fef0a051abf082, 0xbb6359324a5430fd),
        (0x379ca525de290a07, 0x3b10367cab8a544a),
        (0x376e61614232382c, 0x3aca894799ef9e96),
        (0xb6d099267c5285fc, 0xba44a99194b99bec),
        (0xb6cac95a848fd4ed, 0xba4e189da839ed62),
        (0xb6a1589b016c9eb3, 0x3a0eb3a19cff56d9),
    ],
    [
        (0xb6157dc0fbaeeffe, 0x397a750db1884eed),
        (0x3c54d14c77bc1688, 0xbfcdc14ea14e89f9),
        (0x3c2340bd04e7ddab, 0x3f84429fef5b5fbd),
        (0x3c4f19a8f5b137a3, 0x3fa367d7d608e4ba),
        (0xbbf4eb6bddef3531, 0xbf59d6eb2bc49e35),
        (0xbba2f121221c3a28, 0xbf5dc4f991b3db86),
        (0x3bbf047b904254ec, 0x3f1315ec04d6e6bc),
        (0x3b9087e84c57d400, 0x3f0571814a1aa301),
        (0x3b2795eb852a844d, 0xbeba2977fa42f227),
        (0x3b3f92ddd9d5b409, 0xbea1e86423086328),
        (0x3aff19e1c4bddb87, 0x3e54a7b82d41f690),
        (0xbad019797b0c36e0, 0x3e3390660a1bb74c),
        (0x3a8a68e98c9b9a08, 0xbde549e8b3ed12f4),
        (0x3a500e13e664d35d, 0xbdbe32cf77b347a3),
        (0xb9d4a45373f28834, 0x3d6eff58d55a39fb),
        (0x39d2d3ed13f5fe2d, 0x3d415e5c0af749d4),
        (0x39116a2ef1b4cefb, 0xbcf0d67f939f4916),
        (0xb907e51a357adff3, 0xbcbef3f2afde997e),
        (0xb8ba32737d1e38e3, 0x3c6c60ed34505e3e),
        (0x38db347adfaf8e1c, 0x3c360519c76d8628),
        (0x387aa159aedf8fee, 0xbbe324831e6fcd1d),
        (0x38366083503c9cb5, 0xbba99201cdb23c66),
        (0x37de183660131013, 0x3b5512064fe60126),
        (0x37b1a5631a9356cb, 0x3b18d0404e2355af),
        (0x37401bfae4d1c1bd, 0xbac3979694373619),
        (0x36fc444a6f9d50fd, 0xba840b6d6d5adfae),
        (0xb6bab8d9fe8decf4, 0x3a2cebab05fc8474),
        (0xb6643dfbc2e7b96e, 0x39ee6703cd7c47ef),
    ],
    [
        (0xbc5a8046704071bd, 0xbfcc075da85beb4f),
        (0x360798e196eec32e, 0xb971c7c4d6d8ab9e),
        (0xbc53b0c30f3f6ea8, 0x3fbbdeb6ff9f55e1),
        (0x3bef54869c181889, 0xbf661eefb74da882),
        (0xbc20492140a086de, 0xbf8229ea914b846e),
        (0xbbc9a664ae9a2d3a, 0x3f30cbcc6778fd37),
        (0xbbd8078d46437191, 0x3f32aa59f5091f7b),
        (0xbb7b81f3aff076e3, 0xbee1c15d5251ae6a),
        (0x3b707e9f1d7b9d80, 0xbed4583f15abd692),
        (0xbb2cc9b93fea46bb, 0x3e831d151a1284aa),
        (0x3b0aa58bde012b8a, 0x3e6b74e57c226d3c),
        (0x3abe0f5313d3a3df, 0xbe19044f1378d3e4),
        (0xba92c7e5b316fd24, 0xbdf93b1ec7cafe33),
        (0x3a314f797aa7c27a, 0x3da61a4e89bf8438),
        (0x3a2985182535db57, 0x3d80d430f7e78fff),
        (0xb9c88dfe73b74d4c, 0xbd2c3b0a8f7a7b0e),
        (0x39abe65ee3cc9c8d, 0xbd010e773677ab43),
        (0x391b33dc1a20b3ea, 0x3cab59410f7e6f2b),
        (0xb913e4b69baed354, 0x3c7b2e693f06a59f),
        (0x38b855eec84bd050, 0xbc24d1b6b5dc1362),
        (0x3895dc4ab685c9a0, 0xbbf17c6418e3a01d),
        (0xb833fe3056697f69, 0x3b999a3de53bc586),
        (0x37f8f01faf48a793, 0x3b628ca7df1c74e1),
        (0xb77f3f6f64fd7c97, 0xbb09f8ce2cdbd2b7),
        (0xb762fd5da648ce9c, 0xbad083f17385f58f),
        (0x3716906b313f0c25, 0x3a7628fa950b5817),
        (0xb6c02a526d5ba00c, 0x3a38dadf7e4128f1),
        (0xb68876e851c5781b, 0xb9e0755c2cf04c01),
    ],
    [
        (0xb60b222179696176, 0x39614f36ef623dbe),
        (0x3c57ba12cd0fc91d, 0x3fca7022be084d99),
        (0xbc17334559c5149d, 0xbf7c650b6b83109a),
        (0xbc4d0f8f3671311f, 0xbfa163191c30aa62),
        (0x3bbcf7c95031f9af, 0x3f526b045287ddca),
        (0xbbf7b71420f279ed, 0x3f5b17602840abf5),
        (0xbb99a95b35521dab, 0xbf0c0a9cee3c842a),
        (0xbb91505ce3b29b1b, 0xbf03e398cbc472ea),
        (0x3b5125ae0676cb8e, 0x3eb3f35db1ff1b8e),
        (0xbb4e582cdfc3c4cd, 0x3ea0e9b612dbd385),
        (0x3ae57a34be37e069, 0xbe5056babcd9632b),
        (0x3ad18fdd5d5f696f, 0xbe32c1a8c8f963a5),
        (0xb9e38df4d597ab98, 0x3de161b6aa9fde63),
        (0xba4ed163038669b8, 0x3dbd4caa1e0162bc),
        (0xba08f4a604b97cee, 0xbd69fdda132e53b1),
        (0x39dc01e3b41a02ab, 0xbd4101c31e1df223),
        (0xb986207f93b27521, 0x3cecdc25a81e55f2),
        (0x3954737eca211d08, 0x3cbe87025cca1386),
        (0xb90b81b0f636ed32, 0xbc68c602211c726f),
        (0xb8c8869ba08d8915, 0xbc35d4a3a29a8a84),
        (0x3873a38e72663f1d, 0x3be0f3be0d4659ff),
        (0x384c0a8ef8364a36, 0x3ba97a197d5f15ec),
        (0x37d74edace719127, 0xbb52f220ef308891),
        (0xb7a8963ecee526ab, 0xbb18bb20459441c5),
        (0xb76b806269e7e9d0, 0x3ac1a0e6c4af5773),
        (0xb6e3b3a0df8cf60e, 0x3a844a1fb6f8097f),
        (0x36cfc680cfe3679a, 0xba2b974aca95015c),
        (0xb616726934bf9753, 0xb9ec9d1d06dc3c83),
    ],
    [
        (0x3c518b0303bbe2f5, 0x3fc931a5a0ae5aa0),
        (0x36121d64a54c4faf, 0x397d0e55f98c2988),
        (0x3c348c3f5f957879, 0xbfb919c8a3f203fa),
        (0xbc0ef6b166c80d4c, 0x3f602a38da6262a9),
        (0xbc207165718dd6e1, 0x3f807ced48910819),
        (0x3bc68f5ae1c266f9, 0xbf2900f33a00690a),
        (0xbbc8c05ccff0a4b5, 0xbf31278d46fd153c),
        (0xbb6f62aece58d8e5, 0x3edb2595529cf1c7),
        (0xbb731bd584077786, 0x3ed2f7c2d608e0eb),
        (0xbb1f324b3544fc75, 0xbe7e212d2378b576),
        (0xbb005c49aacfd364, 0xbe69f3fcf3638b5a),
        (0xbab2a5d20b39cff5, 0x3e144fbf0377cdfa),
        (0xba636fe0444316b4, 0x3df82268e83f4ed0),
        (0x3a40f4d52865b8bd, 0xbda26cc2b02dc737),
        (0x3a2cea5ea61316c8, 0xbd80418bae8bbc48),
        (0x39b62f3557be1ae9, 0x3d2812815e643e0a),
        (0xb995174e2d5d5173, 0x3d009b2ac849ff66),
        (0xb94bf84e1c96b290, 0xbca7c55d0a6d073d),
        (0xb91bee74f41f6ceb, 0xbc7aa167b939dcf6),
        (0xb894276de305638b, 0x3c2264568716791b),
        (0x389d8028e4fa69b9, 0x3bf137648ca28b1b),
        (0x383829bd6501c349, 0xbb96ed64da9d1ba3),
        (0x37d8257b2a34401a, 0xbb62564f7e37977f),
        (0x37a8ba671e5c18cd, 0x3b0789c67b0265e0),
        (0x377cd4a2067dca1a, 0x3ad05f6c5196ec96),
        (0xb719395c6f8dc2fc, 0xba7443a74dff8e34),
        (0xb6d72ed3f4c7738e, 0xba38b9f499af8cbf),
        (0x366f755619b61063, 0x39de5a75c31562e5),
    ],
    [
        (0x3613f88ad2184341, 0xb97deb1fbadd35dc),
        (0xbc63db68c5672838, 0xbfc80781c32422e7),
        (0xbc1bf83906fd4d0d, 0x3f754eda697a0098),
        (0x3c232419a9d405ca, 0x3f9fbe6df840847f),
        (0x3bd5a76a98bd919a, 0xbf4be318d61276e1),
        (0xbbe633162d1131ea, 0xbf58efee4094379c),
        (0xbb4139e1a56cbd35, 0x3f059145b4f0e4de),
        (0x3b2db153b13f607a, 0x3f0282d26a74c38e),
        (0x3b48c60f3e75a12e, 0xbeaf56c29d9ad959),
        (0xbb2ffef5e5fc81af, 0xbe9fdd03174f902a),
        (0x3acf87aa05cc99f1, 0x3e4a44a7907fee59),
        (0xbaded732e6cb4b4d, 0x3e31df65332ab3e4),
        (0xba6ed0c9b89ddac3, 0xbddc96e9cf361a43),
        (0x3a2bdfd4155ccb1c, 0xbdbc3439f3fac67e),
        (0x3a0882dd9c8f8976, 0x3d65d38ae50afdda),
        (0x39c918a46e201723, 0x3d40833ea1f83461),
        (0xb9805ea372f2e760, 0xbce8afb70fb3c9e7),
        (0xb9589e6fcbbece80, 0xbcbdda411e59ed93),
        (0xb8fe64047a8b5cb6, 0x3c65889665db0701),
        (0x38cc032f117d0608, 0x3c35792846a5cf0c),
        (0x3877212a63ae9eee, 0xbbdde1c2aabb0157),
        (0x38433a64e3a464ef, 0xbba92deda759a7ee),
        (0x37f57ed77143c7c0, 0x3b50e62605143db4),
        (0x37bcf9184fd6279b, 0x3b1889818af5b2f7),
        (0x37585a8626052c83, 0xbabfc6cafa7cab59),
        (0x371544bf1179e875, 0xba8431694dabf284),
        (0xb6c8663570999d11, 0x3a291671421e9a9a),
        (0x366d87c073318aaf, 0x39ec7c2e9f5155fa),
    ],
    [
        (0x3c559364e26d93ec, 0xbfc713fc51664c74),
        (0x35fb44d96564e55b, 0x395783f5327e6c23),
        (0xbc488077fd102b68, 0x3fb7049760cde490),
        (0xbbff945ad33190e5, 0xbf58ef5f1cbe4874),
        (0x3c0b034f6fcde56c, 0xbf7e5f53caf3bead),
        (0x3bcb16596420529e, 0x3f237b0b62ddadd1),
        (0xbb74114140670232, 0x3f2fd3bac08286da),
        (0xbb66bf25df781814, 0xbed5789803de3afe),
        (0xbb30f115b8c5264b, 0xbed1c0faa89993b9),
        (0xbb1173613a405c7a, 0x3e7845b49b0674ba),
        (0x3ada42fc27fec004, 0x3e6886872801396c),
        (0x3a9cf2618301d0ff, 0xbe10b036779ac071),
        (0xba8ed0ec9b47defb, 0xbdf7049d18354c88),
        (0x3a3b54899e1cad3f, 0x3d9edd9d0fb2ebd2),
        (0x3a01a5a29ccc845b, 0x3d7f44600957fa58),
        (0xb9ca58ef04ecc85d, 0xbd2486b1d6a91df8),
        (0xb9990a9ead891a51, 0xbd0016025abd20b3),
        (0x38e7c870470babf1, 0x3ca498b3d1754fe9),
        (0x38f03447a5f61ea1, 0x3c79f47baf518c44),
        (0x38ceb6119f0d50ac, 0xbc2029b685b74cd7),
        (0x38512b942de2f7a8, 0xbbf0ddbdab3fefe6),
        (0x3827e51f6a1999e3, 0x3b94667d2d99941b),
        (0x380b6b9ec305a50e, 0x3b620a8a16336d5f),
        (0xb7951f2acc7ec6bd, 0xbb052c2d5a568fd7),
        (0x377c8c53cd7e3c8e, 0xbad02a6eef6c4f74),
        (0xb70808e163129b9f, 0x3a7266d35702aef4),
        (0x36b4acbcd065af62, 0x3a387c082ca923a1),
        (0x362c2b7e6c7f80f3, 0xb9dbb92bd7a2a5fb),
    ],
    [
        (0xb5fce01036202a2f, 0xb977cba8376f75ad),
        (0xbc6d2f0105f3ce79, 0x3fc62d94d97e859c),
        (0x3bd10ece6e29dfc9, 0xbf70bf614807033c),
        (0xbc37013075a066ff, 0xbf9d5f857a2a6107),
        (0x3be49380eb03f59c, 0x3f46081b0b7fe572),
        (0x3bfce3676d17b413, 0x3f57307b03e248f8),
        (0xbbabc5fb92b99d48, 0xbf0132c0aa83d0dc),
        (0x3b962d899f08a69c, 0xbf0154ed4598d2f0),
        (0x3b436681f34a9d12, 0x3ea94f64f476e615),
        (0xbb2a69d5b518c645, 0x3e9e12725853a9fb),
        (0x3aec161d9a9d14fb, 0xbe4588c758dfc8ab),
        (0xbade3ba8f5e7da28, 0xbe31021cdd9b5f7f),
        (0x3a7b80d1898736a0, 0x3dd7cfa7a6c26b53),
        (0xba5b73ca7f2fd400, 0x3dbb0e011d23f5e3),
        (0x3a0a4b395e402c3a, 0xbd6276cc0fa65a6a),
        (0xb9d2e425ec7c0961, 0xbd3fe92f0882b440),
        (0xb97dfb208f190c67, 0x3ce53126d86bf486),
        (0x39466ab6083aa181, 0x3cbd07f6c1993c8f),
        (0xb8fc1c6cc0e8c2a3, 0xbc62bc249a86f555),
        (0xb8dace7e8fe0ba98, 0xbc34ffb4f078a4b9),
        (0x38784b88dbeaf757, 0x3bda504d640eb072),
        (0xb84627e1aa462991, 0x3ba8bd8d3c086d8d),
        (0xb7acd97ac1b07880, 0xbb4e157f9aadb1fb),
        (0xb770a16ecb1b132a, 0xbb1834ef40762499),
        (0xb6fcba2044990927, 0x3abc8e561ba674cf),
        (0xb7113212416c8ac8, 0x3a83fd2ac64e50dc),
        (0x36c9bcf42c5735bb, 0xba26bc42552c47ab),
        (0xb68c57d3f3c335cf, 0xb9ec3b9a1d42e3e6),
    ],
    [
        (0xbc5b858aeca267e1, 0x3fc56b97f8091ac5),
        (0xb5db386c16851988, 0xb958742bfa32f6c1),
        (0xbc21d2a41bfc4c19, 0xbfb560fcc8c08469),
        (0xbbe6529450b1fd05, 0x3f53fafa39618883),
        (0xbc1ad54776e42e71, 0x3f7c49141623372f),
        (0x3bb39ecd4785a5a4, 0xbf1f69980694fd17),
        (0x3bcc2a33b89df083, 0xbf2dc5f848aa9d34),
        (0xbb7ed412867be351, 0x3ed178fc979b77bd),
        (0x3b73a93300f08ebf, 0x3ed0b494a4bafcc7),
        (0xbb1a1471efdf1226, 0xbe73fc3884c2743a),
        (0x3b0b5eabaae144fe, 0xbe673afb9fb4d844),
        (0xbaa47c72bb33286a, 0x3e0bd90346a3e0ed),
        (0xba99f0845cc41fa4, 0x3df5f3bafb215b44),
        (0x3a2ab1757b26fafb, 0xbd9a1c27c37569c8),
        (0xba09d78a09cb2510, 0xbd7e0455dbf9cc4f),
        (0xb9cc60b0fda73c5a, 0x3d2199b19fcacda0),
        (0x398ae18bb5a4d6cb, 0x3cff1334609dc3bc),
        (0xb93dfbfdb03c0541, 0xbca1e37265d5ce13),
        (0xb9124b9c1211f5ba, 0xbc79353f1aee07d2),
        (0x38b6ddd071a04833, 0x3c1c68e4abc15d44),
        (0xb89af51247543220, 0x3bf075c5531fabc6),
        (0xb83bd38f05e57450, 0xbb921f6f50975ff8),
        (0xb8036a56ab0cdb14, 0xbb61ae48288fd913),
        (0x375b9f05da83eefa, 0x3b02fdf3cf9be443),
        (0x37523bbd552356c6, 0x3acfccedcdaa6c1c),
        (0x370a91e99171a293, 0xba70a685da70f4ee),
        (0x36b74aa0dfffb869, 0xba38280b96abaf7a),
        (0x36763d2c62a700f6, 0x39d93cfbd5a050e0),
    ],
    [
        (0x35ff76e3d2dd1a44, 0x395b36c3bc5679d2),
        (0xbc5e9088e9ff251a, 0xbfc4b2a38f1ab9b4),
        (0x3bf658189763e156, 0x3f6b3878aadeb34d),
        (0xbc3525fe797e2125, 0x3f9b750d89a9b35f),
        (0x3bb2c48be700fe5e, 0xbf41f6911725a956),
        (0xbbed0a355d52c199, 0xbf55beee6fd51c8a),
        (0xbb8079c92f6ce382, 0x3efc3625d7a65089),
        (0xbba1cffb5ca00b9e, 0x3f005375a588a72b),
        (0xbb398a01195f4a0d, 0xbea4ee5e4e7cb18b),
        (0xbb2804ad81294950, 0xbe9c7b3d81b5ff94),
        (0x3aea91d5260903d9, 0x3e41fce14f48518c),
        (0xbadb70b3abcec291, 0x3e30346643c9d86e),
        (0xba7397d52845348b, 0xbdd41c861bc1c34b),
        (0xba5410b5ffe22406, 0xbdb9eeda214eabbe),
        (0x39f8192b2b15935c, 0x3d5f8cf277ce6338),
        (0xb9de17c1ee8e15dd, 0x3d3ec46e14cd5d9e),
        (0xb97a6de668e3d04f, 0xbce2500181b463c6),
        (0x392d839824b54192, 0xbcbc24c2b350e45d),
        (0x38e6e5769440b209, 0x3c605d9c149f1949),
        (0xb8d2d7cef4612fe4, 0x3c34752e1249361d),
        (0xb85cc90966d8e53e, 0xbbd73842b122f4e9),
        (0xb84a80ace8d16161, 0xbba83594fd8e48aa),
        (0x37e69bb04bf99132, 0x3b4acb92dbb1222f),
        (0xb76f84be471103d9, 0x3b17c779363e533d),
        (0x375bd738b69b7baf, 0xbab9a6bc39977c62),
        (0xb72709f09b9c49c1, 0xba83b3d20f9f7463),
        (0xb68fc77a615eb219, 0x3a2494670cd27cf5),
        (0xb6826334c4bccfb0, 0x39ebe0ddc5ab4300),
    ],
    [
        (0xbc4d00ae4313eed0, 0xbfc413644356a52b),
        (0x361469f16306ea73, 0xb97886d88ac06c23),
        (0x3c2e7840863e7c12, 0x3fb40bb88c6f2b85),
        (0x3bd5631193c7bbe6, 0xbf5078d13cfc400e),
        (0x3c1a406bffcdbe30, 0xbf7a9191262ab9d5),
        (0x3bbe2809cb4317d4, 0x3f1a005297618f35),
        (0x3bc92fd90e2b1391, 0x3f2c0cbad847a60e),
        (0x3b6faeaea4946c57, 0xbecd1a72e7c35fd8),
        (0x3b6a2e571e687616, 0xbecf9a2654099c40),
        (0x3b14a1a09cdfd73c, 0x3e70c6b06e20fcbe),
        (0xbb08c15da7f48d8f, 0x3e66136d6425ea74),
        (0xbaa9c15de165ee39, 0xbe07977677c6badc),
        (0x3a91f269e34226c9, 0xbdf4f77b3143a31f),
        (0xba2add845990ae24, 0x3d965720b0a36de3),
        (0x3a0614d5e1fc171d, 0x3d7cd126dc5db8d4),
        (0x39acdeda9cc1b993, 0xbd1e6d575a54f608),
        (0x3998efad7cccf4f0, 0xbcfdfb4d0beacd2e),
        (0xb93dd45bf2db34c1, 0x3c9f3cccbe3a6b23),
        (0xb90d9046de649745, 0x3c786f5c7a82a7c3),
        (0x38bcc387ac92fa4e, 0xbc190b8ec99c1761),
        (0xb897fa896ae9170b, 0xbbf0061c155efe6d),
        (0xb80e81b9d7fb3a38, 0x3b901f2cce805aa0),
        (0xb7e9b68e9f1a61ca, 0x3b61476b1f425669),
        (0xb7a17936a7b35d36, 0xbb0109964cb5ee15),
        (0xb71070e07280a9b5, 0xbacf2f6936a50cd9),
        (0xb704566a8e45b2c1, 0x3a6e1ae6370795ee),
        (0xb6c4223bfe21f7a6, 0x3a37c2c229a343d5),
        (0xb6672d858cefe400, 0xb9d6f33f958fa10d),
    ],
    [
        (0x3611a27c9ec03a6f, 0x39706267910ea7d6),
        (0x3c4997782859a008, 0x3fc37aaceac987b9),
        (0xbc09de167b0f525a, 0xbf66afe4fe0bc0f7),
        (0xbc3d91e80a0529b5, 0xbf99de7a33bc3a97),
        (0x3bdacaad4f266c0b, 0x3f3e024f567ac487),
        (0xbbf5aee36a6b58b3, 0x3f548843c426abe0),
        (0xbb8ed7ac2e5ef1cf, 0xbef7a8e14711c0f5),
        (0x3b9b367577010374, 0xbefeeceb341ad833),
        (0xbb4d96e37263f84b, 0x3ea1a743e05b39ca),
        (0xbb36e8969000eefe, 0x3e9b143d39c90dd5),
        (0xbad319f8c17a1d1b, 0xbe3e8e00012321d3),
        (0x3aae89c89982dfa2, 0xbe2ef28e323e7496),
        (0x3a7f30c6340427ad, 0x3dd137a1bf5bde2c),
        (0x3a5bf7e9721f9c44, 0x3db8e087c7cb7f1d),
        (0x39fb9386784ae3b7, 0xbd5b3dc9dd36d846),
        (0xb9c9416eb664afb4, 0xbd3da6ae03deea14),
        (0x397c9f46130e8018, 0x3cdfe64e83599734),
        (0xb94f05115daba819, 0x3cbb3e899a8275ba),
        (0x38c31d3c51684ee2, 0xbc5cc15471c3a370),
        (0x38dff660cb4ed828, 0xbc33e3495674137f),
        (0x386b19ea37ea8387, 0x3bd491cabca4a282),
        (0xb7fee9491d40343c, 0x3ba7a07e1fc95f06),
        (0x37e4a41a5994c845, 0xbb47ecb9be67f4a7),
        (0x3786f1c46d712a3a, 0xbb174a2d06d7d6cc),
        (0x37594e2ff6a6309a, 0x3ab712a9f6839c0d),
        (0xb7147e96a2d3fad0, 0x3a835bcefeaed781),
        (0xb6c430af3e496108, 0xba22a3254eb3d159),
        (0xb683689f2fc779f3, 0xb9eb727d700d89b6),
    ],
    [
        (0x3c4f123683cf20a8, 0x3fc2f4e70d6c7e01),
        (0x361f442aef336146, 0xb97591bf30ddf0c2),
        (0xbc5702b97938a87d, 0xbfb2ef24d6f7526a),
        (0x3be17c9efd992fc9, 0x3f4bc33c9dc6ec82),
        (0xbc153a673a682d1e, 0x3f7920414ee2acbe),
        (0xbba8c8602e55078d, 0xbf15f9173916a219),
        (0xbbbd01332ea23729, 0xbf2a94fdbdcec471),
        (0xbb6340ead67ee22f, 0x3ec8b309990f950c),
        (0x3b62bfaf2a2b92bf, 0x3ece087ff4517c02),
        (0x3b05bdd16ba2b30c, 0xbe6ca22ab12cb4b7),
        (0x3ad498149d36365c, 0xbe650d1f28635cda),
        (0x3aac3cc8c5cf9b92, 0x3e04441552e259a7),
        (0xba95d08f72147a49, 0x3df411b8a8258957),
        (0xb9f001851befe105, 0xbd9354e914c31b45),
        (0x3a1141eaf70efbe6, 0xbd7bb16f111ebae7),
        (0x39b1abb08be1dbdb, 0x3d1a888536e27439),
        (0xb992bb2516838f25, 0x3cfced726c1026c8),
        (0x392858a15531d1cd, 0xbc9b745d75ff638e),
        (0xb8f68dca35d3026e, 0xbc77aaea63220979),
        (0x38ba5598d094c6db, 0x3c162ef6548b3d48),
        (0x3868e319c4957108, 0x3bef27eac43efa6e),
        (0xb823d9360fee46a7, 0xbb8cc635a6cfb635),
        (0xb801a40c068e9744, 0xbb60db1d947bfc05),
        (0x37881141d0dd48d0, 0x3afea05195871c93),
        (0x3766e70475fe3f97, 0x3ace8489e5ad7b34),
        (0x36ff2a338fa956ef, 0xba6b3d80597f37cb),
        (0xb681333c765d8ec9, 0xba375182dac04c91),
        (0x367a4126c3cf44ba, 0x39d4e0f3b5cdb046),
    ],
    [
        (0x361be200b4a5908e, 0x3974f957842c3586),
        (0x3c6b7326e3fbaa6e, 0xbfc2740819f1caaa),
        (0xbbad91d1969669a0, 0x3f6349369dc780bb),
        (0x3c211249d3675d9f, 0x3f98868d7401bf2e),
        (0x3bab901426ad7111, 0xbf398cd1bebe1445),
        (0x3b8e845cf46e9a08, 0xbf537eef9aadeee2),
        (0xbb978bd723278bd2, 0x3ef43394c95b2d2b),
        (0x3b98d34686b7250a, 0x3efd6dfcdb026028),
        (0xbb39039620a09507, 0xbe9e448fbc8a1f43),
        (0x3b1ab8335f58d6d9, 0xbe99d764ee07c839),
        (0xbad24a385e18b9fd, 0x3e3a53958c8d74ff),
        (0x3aae692fd829adc7, 0x3e2da0e1c8a08d37),
        (0xba68265b22821350, 0xbdcdd7f918de776f),
        (0xba554536fed50ee3, 0xbdb7e68037d5c861),
        (0x39e8fab1857f8e79, 0x3d57c2e63bbe27ca),
        (0xb9b6eea6ea65f06f, 0x3d3c973a175ffbb6),
        (0xb97a4e6705d4e73d, 0xbcdc033788921734),
        (0xb95d7517b81f587e, 0xbcba5d9f2bdd8af4),
        (0x38fc5fbb6c45e1de, 0x3c596cb946e789e8),
        (0x38c497a6d69d23c5, 0x3c335090afbe12d6),
        (0x38799c64a296461a, 0xbbd24f90893c37df),
        (0xb7fbecdd04e88b2a, 0xbba706010539124a),
        (0xb7eafe8f4b0c9808, 0x3b457032af41d7a4),
        (0xb7bc178a723aab4c, 0x3b16c43d5e0cacac),
        (0x37373a6759c668d6, 0xbab4ce42861296d9),
        (0x3714ea53a56cccdf, 0xba82fa99ffa503c9),
        (0x36a6206269dd287a, 0x3a20e7e4d182d7e6),
        (0xb64ef30807ed040a, 0x39eaf6aa0936f5c1),
    ],
    [
        (0x3c34f78a7cfd8a8a, 0xbfc20198200b699d),
        (0x360565ac287f5dbb, 0x3964862986ce6868),
        (0x3c1ede75d6e0c0a5, 0x3fb1fd242a74e630),
        (0x3be31a446a06705c, 0xbf47cf261dfbf19a),
        (0xbc0b11e38b0823ed, 0xbf77e4820ec1dde4),
        (0xbbbbba7e6db236bb, 0x3f12e1bd281dfcbb),
        (0x3bc48aed48d0a238, 0x3f2950bb06c6fdf9),
        (0x3b5f50566d2c1e7b, 0xbec54a38ab6af546),
        (0xbb69fdc0d6c239ec, 0xbecca94f38024fdc),
        (0xbaf76dde5126de40, 0x3e68c7e75971c85c),
        (0xbb0678751bf4f650, 0x3e6423fc7e251ca9),
        (0x3aab4294dcf0c897, 0xbe019fe1d8e51b96),
        (0xba976b8065979a15, 0xbdf34198c795142c),
        (0x3a1b7155c351ab8b, 0x3d90e78cdab26dd9),
        (0xba1a3a4b53edd6ed, 0x3d7aa74cb5ba4b0e),
        (0xb9a9e4993f6f7c4a, 0xbd1756ee8369797d),
        (0xb992ea1492b57e10, 0xbcfbee3ccbac62a3),
        (0xb9058529ec2e805b, 0x3c984c93627c1378),
        (0xb90c3b9cf073d667, 0x3c76ecd2977e72a5),
        (0x38be2449e2bf487c, 0xbc13c1ad21981036),
        (0x3883f6d9af30b491, 0xbbee45c90965f9de),
        (0xb7f83349481ced2f, 0x3b89c8e8d181a236),
        (0x380803a9275b7f65, 0x3b606d491b5da420),
        (0x3798109188279175, 0xbafb9c064d80a138),
        (0x376013fe3b9f77b2, 0xbacdd315dbfb672b),
        (0x36fc9d61cb3c3abf, 0x3a68b2de73775fa8),
        (0xb6ce777cbee59127, 0x3a36d90ef88dab95),
        (0xb67862ced9ec7588, 0xb9d305b1d99a3c61),
    ],
    [
        (0xb5fb86fee5ebbcf6, 0x396f19a82c6866ac),
        (0xbc4081c2a50ad27f, 0x3fc192f2627a74e3),
        (0x3c06268e5916d632, 0xbf60a846a83fecf2),
        (0xbc1be434e30d7a5a, 0xbf975eceaabf7f86),
        (0x3bbc2d7f1eb154d6, 0x3f3617c581be35b1),
        (0xbbb0beeca29a2baa, 0x3f529934b7a84483),
        (0xbb91b58b66aab0a7, 0xbef18123e875188a),
        (0x3b94740ce7aaa9dc, 0xbefc1f05a2d85165),
        (0xbb3fd873b63ab5e9, 0x3e9a4e0bc0926545),
        (0xbb233d21389bce0f, 0x3e98be81ad44d933),
        (0xbabb1f9ca258cce3, 0xbe36f73795e25c01),
        (0x3ababafde69f161c, 0xbe2c70ab158cd19a),
        (0x3a5796bd01bf9720, 0x3dca262190472d9a),
        (0x3a444e2c448f52df, 0x3db70112aab52bc2),
        (0xb9f5a65a43d3cb6e, 0xbd54ec163f88c02f),
        (0x39d52f72c43ac9e8, 0xbd3b991575a9daff),
        (0xb9742584b2344dbc, 0x3cd8cbabadfd45c4),
        (0xb952f19a04e89b00, 0x3cb9867f1ea85270),
        (0xb8f407ccf94d1457, 0xbc56a0c58ce961b7),
        (0xb8d0795ea11bd36d, 0xbc32c10fa1783b9d),
        (0xb85c7f401bf9086f, 0x3bd0630454542422),
        (0x3831adbad02f8547, 0x3ba66b5f3f83dfb5),
        (0xb7c18a605d4d2201, 0xbb434a75d1b13844),
        (0x37a6e2dd153188df, 0xbb163af22779791d),
        (0xb75d10a8ae54e7d5, 0x3ab2d2655ffc3033),
        (0xb72fe47a60aa5d90, 0x3a829478c1a243e5),
        (0xb69af79b5351dbd0, 0xba1ebe95a9db6af0),
        (0xb68a3c131c1f2df8, 0xb9ea729d3347c622),
    ],
    [
        (0x3c68f71f103b6bf8, 0x3fc12f9870d68e18),
        (0x35eedd0c814b660b, 0x39584c0de5cb23f9),
        (0xbc5fd0d25668ad6b, 0xbfb12c11811945f9),
        (0x3bebc70ffc015992, 0x3f44b638f21f0f76),
        (0x3c1bc10a4e0506fd, 0x3f76d2a897d58353),
        (0x3ba5de76c7b42d7a, 0xbf10732e5458ba20),
        (0x3bcfca03362738fd, 0xbf2835929300df3f),
        (0x3b4992f62f114659, 0x3ec297283816a83b),
        (0xbb50d6dea52a7094, 0x3ecb73adedf11a43),
        (0xbb0dce80931dc602, 0xbe65b455b903f09c),
        (0x3b0295728813e0af, 0xbe6353f0797a6699),
        (0xba8bf23e761207b9, 0x3dfefc9ac17b1eeb),
        (0xba878c04e9d38305, 0x3df28535463ce067),
        (0xba096e621beb197b, 0xbd8dd894cbbee3f9),
        (0xba0f45ad9ade804f, 0xbd79b288cc392b5b),
        (0x39916204abd42d65, 0x3d14b2ec9ace070c),
        (0xb9936d340e28fba5, 0x3cfaff72f6057831),
        (0x393a603e53eacc06, 0xbc95a7bfada0275b),
        (0xb901b469e17f5bb3, 0xbc7637b7b45acfc3),
        (0xb8820547fceb973f, 0x3c11b216f1fbcae9),
        (0x38886047cf777ac4, 0x3bed6a4aa2540e36),
        (0xb7deebf62d61db26, 0xbb87366125ff8ead),
        (0xb7f352b13b569959, 0xbb6000a238ddbac3),
        (0x378f208cf7576478, 0x3af8faebb4eb88ac),
        (0xb7338292ec57b7e3, 0x3acd201e6cbbf759),
        (0xb70a1ecbbc05a2ed, 0xba6674d821ccdd7a),
        (0x36c727a7e3d4fb6c, 0xba365d344fa9d6d7),
        (0x365f47ff2787d3f1, 0x39d15e5b7cf8fc11),
    ],
    [
        (0x360bdb5b805e25b0, 0x3969e1daf83188a7),
        (0x3c60c06e2860e867, 0xbfc0cf3ee98f769b),
        (0xbbfc2c390327e76c, 0x3f5d26e7af251f79),
        (0xbc394225ed089993, 0x3f965d05948a946a),
        (0xbbcb3460aae307ce, 0xbf335959b8482e40),
        (0xbbf744279fabe84c, 0xbf51cff175d05c2a),
        (0xbb83714c1d2b527e, 0x3eeeb59416879106),
        (0xbb89158b4542d28b, 0x3efaf7544eeac766),
        (0x3b284917e9916dd8, 0xbe9720522bb1fc81),
        (0xbb31671693e89a81, 0xbe97c41261705419),
        (0x3a886ab5cb454f45, 0x3e343fa0ea5e007d),
        (0xbab8f90f9f428048, 0x3e2b5e23abebd7a4),
        (0x3a64e5d5f577d2d2, 0xbdc722397e750312),
        (0xba51768fabb7fdb1, 0xbdb62f2174e6be49),
        (0xb9fba4dd6a628798, 0x3d5294ab060a163a),
        (0xb9de8957f248872f, 0x3d3aace944141c61),
        (0x394ff7327b4ba565, 0xbcd61cc8146cdfa1),
        (0xb9394e82e72782f6, 0xbcb8bb3c270a284d),
        (0x38f4722936097c3e, 0x3c54449d966478fa),
        (0x385797f171c71fea, 0x3c323712fdd32aac),
        (0xb85151bbb56468ad, 0xbbcd7cedd0538c51),
        (0x3800563232817820, 0xbba5d3f2f4d3b9e1),
        (0xb7d9661fbc11e8a2, 0x3b416f9a7e4f8d43),
        (0xb7b5912190425469, 0x3b15b1f1d8d05a13),
        (0x375dae4cd5158605, 0xbab116c0b74e863e),
        (0xb72f2d5922ef51d5, 0xba822c909d7ff79e),
        (0x3686e084ffb6066d, 0x3a1c0950bfbf78de),
        (0x36899d263bfc66cf, 0x39e9ea82b7b42ba3),
    ],
    [
        (0xbc6742aa46ea9b31, 0xbfc077eede4a0d89),
        (0xb5d51339f008438b, 0xb93d0e3fbdb78328),
        (0x3c3cabc5883fc763, 0x3fb0751548b2924d),
        (0x3bd8c3b817f5be4f, 0xbf423b5d46a73864),
        (0x3c1e7b0b0210cbce, 0xbf75e2467c8fb832),
        (0xbb9abdc39961759f, 0x3f0cfe5c189d6e4d),
        (0xbbcf4c0c63a83958, 0x3f273bbd8c7aef2d),
        (0x3b6d9272577eb3fa, 0xbec06974d3d04287),
        (0xbb56fb637948d35e, 0xbeca6081d36e6a2b),
        (0xbaee9b9b2585b7c5, 0x3e6334a83cf6092b),
        (0xbb0e3539c25f29f0, 0x3e6299571cb4e1fb),
        (0xba80b2cd9f616718, 0xbdfb7f5bc0aada50),
        (0xba818316569b1592, 0xbdf1da63b4d6f878),
        (0xba03f4308aff6abe, 0x3d8a9288ceccd9f2),
        (0x3a16f5c0a511bfb0, 0x3d78d1d74c88ee4c),
        (0x39a16735a5eb7383, 0xbd127eed3a482d7f),
        (0xb9883ce9962db86a, 0xbcfa2143706ccb78),
        (0xb933e39af0b0a816, 0x3c936d1464c4bf22),
        (0x390afef7e49ce49a, 0x3c758cc3ad07d84d),
        (0xb8ae01cde6f904a8, 0xbc0fe0c8ce96de05),
        (0x387fd4921dadf9fd, 0xbbec97f7854a0777),
        (0x37f77fd4b81374ba, 0x3b84fee62588d5e1),
        (0xb7f6a3ad2cef6d22, 0x3b5f2dcbb8b70749),
        (0x379518518eb791db, 0xbaf6b09656bd01ca),
        (0x3755aca529972d2e, 0xbacc6f2f1ddb116e),
        (0xb70b72b0eec38d9d, 0x3a647b8c438cfbcc),
        (0x36cc4b3943e41f19, 0x3a35e0cb3264bc18),
        (0x36290f8597bf8fe9, 0xb9cfcd0aefeb1cbe),
    ],
    [
        (0x35d2722913fb4c63, 0x3944e3fd926f8fb1),
        (0x3c61166b7995967a, 0x3fc0230ba90f2871),
        (0x3bfb746b17280a35, 0xbf59ca16f0c9734e),
        (0xbc375b99fcae0fa4, 0xbf9579c1bdbcfc99),
        (0x3bc1259c7efa1bb2, 0x3f3120ecfac5c017),
        (0xbbd35a76e06cda5a, 0x3f511dd26bbe2946),
        (0x3b72ed405d5bc408, 0xbeeb37e7c9a57149),
        (0xbb86e93d7c515f2f, 0xbef9f01e7c1909a0),
        (0xbb2f562d62e74ecf, 0x3e94887fe7a88c2c),
        (0xbad010f8761951e4, 0x3e96e37238841d4b),
        (0xba99b7f17d43bc36, 0xbe3204b644d6a04b),
        (0xbacc8fcda9cad56e, 0xbe2a659b13eebf8a),
        (0x3a42b7267c5e128a, 0x3dc4a40c87601c97),
        (0xba57720418c52447, 0x3db56f0250e4fcac),
        (0xb9f80ede07f2b143, 0xbd50a1011bf4a7dd),
        (0x39d51971dde44362, 0xbd39d22a6dc5a57c),
        (0x3973c4be23c7c005, 0x3cd3db0f19da31d9),
        (0xb95d91267d6c60a3, 0x3cb7fc7d4070961c),
        (0xb8e43885990b2b03, 0xbc5243aa7bbc7ede),
        (0x38d02229cd94d6a5, 0xbc31b3c3dd0d78fd),
        (0xb83af6a3d8823899, 0x3bcaabdcab96df5c),
        (0x384894bf3b4da9f0, 0x3ba541bb402c78ac),
        (0xb7dda9ab25eaff8c, 0xbb3fa915b2fa47b8),
        (0x379cbdf7cf876dad, 0xbb152ba169685593),
        (0x3733162e8c1ee94b, 0x3aaf26066ae9706e),
        (0x36f9819114677bb3, 0x3a81c51763321aee),
        (0x36a446eca6c003a9, 0xba19a5c3984013f5),
        (0xb68b1567ca85f2fe, 0xb9e96168a239b1d9),
    ],
    [
        (0xbc446154ab44acb3, 0x3fbfab0b166d23d8),
        (0x35dbf46a8ed0ff76, 0x393d7c0296d587f1),
        (0x3c4290cc8b89531c, 0xbfafa65c1ce7ebd6),
        (0x3be9bdd7d19e6706, 0x3f4035bf503ffc1f),
        (0x3be801473cc85f34, 0x3f750d1b04713c41),
        (0x3babe22a416a834b, 0xbf09cd14a92842a2),
        (0x3bca868e6f4e4834, 0xbf265d504af5d8fe),
        (0x3b5814b5b96336a9, 0x3ebd3feeb33d9cee),
        (0xbb4c05f6b4f495e3, 0x3ec96a257062f76e),
        (0x3b0b04fde267ccf6, 0xbe61254f302b36b2),
        (0x3ad771b2683ddadc, 0xbe61f11585e04fca),
        (0x3a8b3d53dc0784a1, 0x3df89a7674dd98cd),
        (0x3a7b6da4a2d7c8b5, 0x3df13f0ba48c1692),
        (0x3a248eb19153b909, 0xbd87d67b47fd268e),
        (0x3a0aa35dfad0cb5d, 0xbd780381812ad9b0),
        (0xb97e8eabaacf7749, 0x3d10a41213b1209a),
        (0xb9694e4f19ae8248, 0x3cf9530714f8af89),
        (0xb935bd4a759fcd07, 0xbc918884a9ff47a4),
        (0x390c5a7313603d8b, 0xbc74ec3ce9f28ddc),
        (0x3891396752d2f4be, 0x3c0cde4ce08c9c88),
        (0xb88f3ed08df90975, 0x3bebd00bdad9a3b4),
        (0xb82845f2323731b8, 0xbb831454d10def31),
        (0x37f8ae176f7e036c, 0xbb5e62383fd8bdbf),
        (0xb7891c6860574ef2, 0x3af4b117f31a9f40),
        (0xb76e1cfee2b9a0d7, 0x3acbc29e8f1ce304),
        (0xb6d4fcd41e5c1c66, 0xba62bec668d1d14b),
        (0xb69b898b21726f67, 0xba3565d792f0e519),
        (0xb667b0601d949d31, 0x39cd328695f5c7be),
    ],
    [
        (0xb5856985dc45fea1, 0xb96fbca69975e716),
        (0x3c50db2c50623ec1, 0xbfbf13fb0c0e6fcd),
        (0x3bf63fd50ee5bfaf, 0x3f5706ed3d935d00),
        (0x3c3a7c91ef9a7d9f, 0x3f94af74cbd77bef),
        (0xbbac31e3b8ee60ac, 0xbf2e9a9e66e5a792),
        (0x3bf35ae9733a6eb9, 0xbf507ec9ed824fcb),
        (0xbb4a161032086ff0, 0x3ee856d4518ab29e),
        (0xbb9af9c37c81cff9, 0x3ef9040de830649e),
        (0x3af6ec0adcd73b17, 0xbe9262f69c56c652),
        (0xbb34da0db90f8b57, 0xbe9618c94a54734d),
        (0x3acfc6bc54a1c19e, 0x3e3029d2c8bf70d6),
        (0x3ab60f30dc12e39e, 0x3e2983bca0a3e67c),
        (0x3a60395cd0053820, 0xbdc28e2a00002a82),
        (0xba51c63c15e96dac, 0xbdb4beeacafc4fad),
        (0x39ed5a030c6605c2, 0x3d4df880e37a5060),
        (0x39d2026270de9105, 0x3d3907be43054aec),
        (0xb952832db501353a, 0xbcd1f10ff685083c),
        (0x395f463c7798955d, 0xbcb74a1dc71abca5),
        (0x38e7d8e5ae99ce88, 0x3c508d41117c90f3),
        (0xb8d552c759edc37c, 0x3c313795880a77fc),
        (0x386b16755d17fac8, 0xbbc83ea1ceb16a07),
        (0xb848188a8a68517f, 0xbba4b5cc8ded0fc2),
        (0xb7d762214edd8666, 0x3b3cdf000bd6f203),
        (0xb789ca564a28d131, 0x3b14a97cac9def65),
        (0xb74e5064a15a2c78, 0xbaac7eef42820297),
        (0x371d86a43d3d4298, 0xba815f89d5c7d990),
        (0xb6ba98e94c93cb57, 0x3a1789ebfe3c8e8c),
        (0xb68a2c469d8fa268, 0x39e8d984b79cb9e9),
    ],
    [
        (0xbc5bf01bf61d8ed3, 0xbfbe891b327da16d),
        (0xb5cc3c81dba6f81d, 0x3953aec892af6d03),
        (0xbc3636379e4f582f, 0x3fae8533ce07bdb8),
        (0x3bc53b9dd0e1816e, 0xbf3d1253218e31b0),
        (0xbc1eae521e3467c6, 0xbf744e6826476498),
        (0xbb6864ec60e7922a, 0x3f07271a9b5e3cb4),
        (0x3bace83941ef46b5, 0x3f2595b697c8ec04),
        (0xbb5f57ab9661648d, 0xbeba46b03ecb38cc),
        (0xbb4dc38fda907530, 0xbec88c173e07621e),
        (0x3add237c79f6db0f, 0x3e5ed9b1754fbd3c),
        (0xbb0d6748bb472b61, 0x3e615891ef314e18),
        (0x3a8caec9a3fbb1e1, 0xbdf62ca352cf904e),
        (0x3a7916cf164a75f3, 0xbdf0b14767c2b010),
        (0x3a2595ab6e7a4beb, 0x3d85879bb4a2a8ac),
        (0xb9f9a36051cbdc81, 0x3d7745bbca5b0e82),
        (0xb9a0856355ee0f2f, 0xbd0e21736ab8469f),
        (0x3985ce3b693f20e7, 0xbcf893b256068d4d),
        (0x3920359c32c5de22, 0x3c8fd42c80d03657),
        (0xb9029a5c2c268edd, 0x3c7455e80611a14d),
        (0xb8a84439970d2016, 0xbc0a4682da82870e),
        (0x3879596804aa9b9c, 0xbbeb12f0bb307616),
        (0xb815b3814c55d4ca, 0x3b816a7f58b7a414),
        (0xb7f11e258570c30d, 0x3b5d9fa088a0baf3),
        (0xb78f1b32689b744c, 0xbaf2f1ad94886e01),
        (0x376c31c2ebc43a82, 0xbacb1bde877c7dfa),
        (0x36f7154043b459d4, 0x3a6136b97f99fb4d),
        (0x3693894e8fe8da2f, 0x3a34edb643b11559),
        (0x36486be2c3a76600, 0xb9cae2d7b5d6550c),
    ],
    [
        (0xb5e117018bd0a4d7, 0x396cc7fef85a11ce),
        (0xbc3b9f1d130797b4, 0x3fbe018dac1c17e3),
        (0x3be072789a66c0ad, 0xbf54b994dd05c1fb),
        (0xbc338b4f0d3a410f, 0xbf93f9e0db07e7ef),
        (0xbb9a8995613210dd, 0x3f2b8e55b75b13ab),
        (0xbbd251fa83bbf995, 0x3f4fdf68a78bb3d2),
        (0xbb783beff5b1e008, 0xbee5ee9d17106a08),
        (0x3b70935ad535759f, 0xbef82ee6dfdfedeb),
        (0xbb36eeb8b4349e83, 0x3e90962d7f6d61a6),
        (0x3b3ddb699c12a377, 0x3e9560edce7d682d),
        (0xbaadbe6efa7bc4c2, 0xbe2d34381d02bdbe),
        (0xbab5faefe93fce4d, 0xbe28b5a0e74ae0d3),
        (0xba651c2d1b2cfd90, 0x3dc0cab953f05720),
        (0xba4cdf9f9b970a3b, 0x3db41d21c68d7ed3),
        (0x39efe10ea47be16f, 0xbd4b2d2dd7cb3c52),
        (0xb9d985842dbfa533, 0xbd384c5369d2e33d),
        (0x396e2573c81613c3, 0x3cd04e17ab25ece8),
        (0x3940629f2b3df4a6, 0x3cb6a38e0aef9402),
        (0xb8e1d69808a03084, 0xbc4e28099fc922c0),
        (0xb8c82332ebaf3236, 0xbc30c28e6023fb30),
        (0x386b9c9fc0c05d07, 0x3bc62487322c4643),
        (0xb84bcacb7b5c3520, 0x3ba430a24f213ff8),
        (0xb7c5039f9a7e3931, 0xbb3a7037139e97ed),
        (0x379c58e611949e56, 0xbb142c5d039a4a70),
        (0x3727db4e84bd36dc, 0x3aaa2a8a29e14a1d),
        (0x36cc215ba322ceb2, 0x3a80fcdc0bece4f5),
        (0xb6b57109b2328425, 0xba15ac88c96a9a92),
        (0x3661fe5c16d66355, 0xb9e854529bb11eca),
    ],
    [
        (0xbc4d7a7b8d722118, 0x3fbd84391bb2748d),
        (0xb5f89e9b40c23cab, 0x39715c553323e555),
        (0x3c4b439a88ed803b, 0xbfad80edb3c4ea05),
        (0xbbd902d343596722, 0x3f3a431f5421f7ef),
        (0xbc13a66c5c4dc379, 0x3f73a282fe7b63a8),
        (0x3bacf6f60b035a27, 0xbf04ed653e607bdd),
        (0xbbb378680e9561b7, 0xbf24e15832bda3c4),
        (0xbb5464c53ca4420f, 0x3eb7c5832dd1374f),
        (0x3b40f0848ee6fe3d, 0x3ec7c2b80da8df98),
        (0xbadb963cfcbd4c8f, 0xbe5bf160ccd35445),
        (0xbac9dbc6af458996, 0xbe60cda2141b2a1e),
        (0x3a876693420f17fa, 0x3df41d31d2a419e4),
        (0xba820602dcfc6a1c, 0x3df02f6ca850168a),
        (0xba125360d2345a18, 0xbd83901f61054085),
        (0xba17df49fb76cb0e, 0xbd7696ce2b748d02),
        (0x39af2c4622bc54e9, 0x3d0b6e82997a390e),
        (0x399ac81facf7b5ab, 0x3cf7e214796bec18),
        (0x3927c686b63cfcd2, 0xbc8d0a3822eede5d),
        (0xb9116570c5a93cb1, 0xbc73c944a2b7a866),
        (0xb8a7e7acb8f1f32c, 0x3c080748c2fd703d),
        (0xb87cb3127b6df48b, 0x3bea608ebfae056a),
        (0xb81b4f7b9cda848d, 0xbb7fee483d8cb2bd),
        (0xb7fcfe24d11f96d0, 0xbb5ce67574b83792),
        (0x3757b86fe74315fe, 0x3af168f6128b62f7),
        (0xb76886b34e3ecd6f, 0x3aca7bbe8016f1c0),
        (0x36d81a3f26ccf62a, 0xba5fb8acc96dbd9b),
        (0xb6dd8ec78e1a519f, 0xba347945efb0e5b5),
        (0xb650c6a09f8b3be4, 0x39c8d517ce0975bd),
    ],
    [
        (0xb5f1cdfb0cbbbf9b, 0x3963ad355553c663),
        (0xbc56edd809f4ec44, 0xbfbd09b21e36c0bd),
        (0xbbfd63853cd12a08, 0x3f52c84acfb586b4),
        (0x3c340975d5184af7, 0x3f9355b904fbf7ee),
        (0xbbc2e53f276aead9, 0xbf28fb570465af0d),
        (0x3bdedf2b372334a8, 0xbf4edc3292ba6cfd),
        (0xbb81b7d6d1c235c3, 0x3ee3e552ee8c2577),
        (0xbb9b6ea46d6db745, 0x3ef76d44f6a83523),
        (0x3b2fdb93f3cf1ba4, 0xbe8e1ee2dc4a3bda),
        (0x3b3cccd425113a57, 0xbe94b944bbd4c7bf),
        (0xbac4687392bdbcfe, 0x3e2a8c3a4ce36f16),
        (0x3ac2a4a12febf7d8, 0x3e27f8c9ef47068e),
        (0xba40ac973313c53b, 0xbdbe9295749ee471),
        (0x3a116490085933e1, 0xbdb38813c3eb77e4),
        (0x39e5a2d8c5b27290, 0x3d48c7cda44b7754),
        (0xb9a05a73d2dc3c66, 0x3d379e90468589ab),
        (0x3961ad36dfeb19e3, 0xbccdca323fb2ed12),
        (0xb95c95ee628cae05, 0xbcb6080be2ba756c),
        (0x38e14a5e61e5df84, 0x3c4b9a8e33ba505f),
        (0x38869b43fbbf40af, 0x3c3054766d105227),
        (0x3869a58ba74326c8, 0xbbc44fda8cffeefb),
        (0xb80e3c951291e0af, 0xbba3b257514f6e67),
        (0x37d7232bb9cc06e3, 0x3b384ea2f19dc085),
        (0xb7b20d806734b932, 0x3b13b4adeb19bbff),
        (0x37470ea8b4edb81d, 0xbaa81cf876394534),
        (0x372493d295b38feb, 0xba809da078876581),
        (0xb6b9d7d8c4984442, 0x3a14055384d957ac),
        (0xb68f3f22c46d785d, 0x39e7d2d65aaf6d5b),
    ],
    [
        (0xbc579e384ae35818, 0xbfbc97d79918527d),
        (0x35eb9172f9b7276c, 0x395ce1b629f3b194),
        (0x3c415f3ee939a0c0, 0x3fac95081ab2b511),
        (0x3bd3c58af0675aab, 0xbf37e0b14f7d7c3f),
        (0x3c167e93b7727b22, 0xbf730688f6836a76),
        (0xbb9eaad91947069e, 0x3f030941f6e78e36),
        (0x3bb70db1e3b30ff6, 0x3f243d5898657a6f),
        (0xbb39f80469778551, 0xbeb5a39a94f2ad54),
        (0xbb5db472078111de, 0xbec70b18406146f7),
        (0x3af60534ae337487, 0x3e597607f9532274),
        (0xbad7032885ed6fc7, 0x3e604e788f4f63b9),
        (0xba876faf3495b1f5, 0xbdf2598919473e7e),
        (0xba64011e20bbe632, 0xbdef70154273be01),
        (0xba12fbde85806234, 0x3d81df5c1ca3869c),
        (0x3a11557ac79b4968, 0x3d75f5256665e918),
        (0xb96dc72056cbaf44, 0xbd091a176f7295bc),
        (0xb99ab0535f46bdf0, 0xbcf73cf989310b2c),
        (0x392f229ecb648d6b, 0x3c8a9f0a04e76a3b),
        (0x38dd6e9f2d0c2bf9, 0x3c7345b205781b7b),
        (0x38a22edb9f6d2f0c, 0xbc0611e4c0bc23e9),
        (0xb888702d77aa4448, 0xbbe9b88423f59f05),
        (0xb7df03c936a4e4a6, 0x3b7d637385b6b500),
        (0xb7f31f276ec1d4a1, 0x3b5c36bdc90c0d20),
        (0x37908786cc7f97a2, 0xbaf00eeea5fbf4a4),
        (0xb7521f3b34bd9065, 0xbac9e29e4e7fe131),
        (0xb6d9c8943eaa0627, 0x3a5d56f0d36bcfd3),
        (0x36cd064c8625bd4e, 0x3a3408faa535443a),
        (0xb66864ee592be672, 0xb9c8d8880a8fa4ae),
    ],
];
