/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var proxyquire = require( 'proxyquire' );
var ENV = require( '@stdlib/process/env' );
var noop = require( './../../noop' );
var nextTick = require( './../../next-tick' );
var exec = require( './../lib/node/worker/exec.js' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof exec, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns a child process', function test( t ) {
	var child;
	var exec;
	var pid;

	exec = proxyquire( './../lib/node/worker/exec.js', {
		'child_process': {
			'execFile': execFile
		}
	});

	pid = 12345;
	child = exec();

	t.equal( child.pid, pid, 'returns child process' );
	t.end();

	function execFile() {
		return {
			'pid': pid
		};
	}
});

tape( 'the function calls a callback when a script finishes executing', function test( t ) {
	var script;
	var exec;
	var pid;

	exec = proxyquire( './../lib/node/worker/exec.js', {
		'child_process': {
			'execFile': execFile
		},
		'./process.js': {
			'env': ENV,
			'stdout': {
				'write': noop
			},
			'stderr': {
				'write': noop
			}
		}
	});

	script = '/a/b/c.js';
	pid = 12345;
	exec( 'node', script, done );

	function execFile( cmd, args, opts, clbk ) {
		function onTick() {
			clbk();
		}
		nextTick( onTick );
		return {
			'pid': pid
		};
	}

	function done( error, id, key ) {
		if ( error ) {
			t.ok( false, error.message );
		}
		t.equal( id, pid, 'returns pid' );
		t.equal( key, script, 'returns script name' );
		t.end();
	}
});

tape( 'the function calls a callback with an error if an error is encountered when executing a script', function test( t ) {
	var script;
	var exec;
	var pid;

	exec = proxyquire( './../lib/node/worker/exec.js', {
		'child_process': {
			'execFile': execFile
		}
	});

	script = '/a/b/c.js';
	pid = 12345;
	exec( 'node', script, done );

	function execFile( cmd, args, opts, clbk ) {
		function onTick() {
			clbk( new Error( 'beep' ) );
		}
		nextTick( onTick );
		return {
			'pid': pid
		};
	}

	function done( error ) {
		t.ok( error, 'returns an error' );
		t.end();
	}
});

tape( 'if the `WORKER_UID` environment is set, the function sets the child process `uid`', function test( t ) {
	var script;
	var exec;

	exec = proxyquire( './../lib/node/worker/exec.js', {
		'child_process': {
			'execFile': execFile
		},
		'./process.js': {
			'env': {
				'WORKER_UID': '123'
			}
		}
	});

	script = '/a/b/c.js';
	exec( 'node', script, noop );

	t.end();

	function execFile( cmd, args, opts ) {
		t.equal( opts.uid, 123, 'sets uid' );
		return {
			'pid': 12345
		};
	}
});

tape( 'if the `WORKER_GID` environment is set, the function sets the child process `gid`', function test( t ) {
	var script;
	var exec;

	exec = proxyquire( './../lib/node/worker/exec.js', {
		'child_process': {
			'execFile': execFile
		},
		'./process.js': {
			'env': {
				'WORKER_GID': '1234'
			}
		}
	});

	script = '/a/b/c.js';
	exec( 'node', script, noop );

	t.end();

	function execFile( cmd, args, opts ) {
		t.equal( opts.gid, 1234, 'sets gid' );
		return {
			'pid': 12345
		};
	}
});
